/***************************************************************************
**
**  This file is part of gptarget.
**
**  gptarget is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gptarget is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-05-28
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <DinverCore.h>
#include <DinverDCCore.h>
#include <QGpCoreTools.h>
#include <QGpCoreWave.h>

#include "TargetAdd.h"

/*!
  \class TargetAdd TargetAdd.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

TargetAdd::TargetAdd()
{
  _type=Undefined;
  _position=0;
  _weight=1.0;
  _minimumMisfit=0.0;
}

TargetAdd::~TargetAdd()
{
  delete _position;
}

/*!
  Description still missing
*/
bool TargetAdd::setOptions(int& argc, char ** argv)
{
  int j=1;
  for(int i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-dispersion-rayleigh") {
        _mode.setPolarization(Mode::Rayleigh);
        CoreApplication::checkOptionArg(i, argc, argv);
        _mode.setIndex(CoreApplication::toInt(i, i-1, argv));
        _type=Dispersion;
      } else if(arg=="-ellipticity") {
        _mode.setPolarization(Mode::Rayleigh);
        _mode.setValue(Mode::Signed);
        CoreApplication::checkOptionArg(i, argc, argv);
        _mode.setIndex(CoreApplication::toInt(i, i-1, argv));
        _type=EllipticityCurve;
      } else if(arg=="-abs-ellipticity") {
        _mode.setPolarization(Mode::Rayleigh);
        _mode.setValue(Mode::Absolute);
        CoreApplication::checkOptionArg(i, argc, argv);
        _mode.setIndex(CoreApplication::toInt(i, i-1, argv));
        _type=EllipticityCurve;
      } else if(arg=="-dispersion-love") {
        _mode.setPolarization(Mode::Love);
        CoreApplication::checkOptionArg(i, argc, argv);
        _mode.setIndex(CoreApplication::toInt(i, i-1, argv));
        _type=Dispersion;
      } else if(arg=="-group") {
        _mode.setSlowness(Mode::Group);
      } else if(arg=="-mt") {
        _type=MagnetoTelluric;
      } else if(arg=="-p") {
        CoreApplication::checkOptionArg(i, argc, argv);
        if(!_position) {
          _position=new Point;
        }
        if(!_position->fromString(QString(argv[i]))) {
          App::log(tr("gptarget: error setting position from \"%1\"\n").arg(argv[i]) );
          return false;
        }
      } else if(arg=="-weight") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _weight=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-min-misfit") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _minimumMisfit=CoreApplication::toDouble(i, i-1, argv);
      } else {
        App::log(tr("gptarget: bad option %1, see -help\n").arg(argv[i]) );
        return false;
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j < argc) {
    argv[j]=nullptr;
    argc=j;
  }
  if(argc!=2) {
    App::log(tr("gptarget: missing target file, see -help\n") );
    return false;
  }
  setFileName(argv[1]);
  return true;
}

/*!
  Description still missing
*/
int TargetAdd::execute()
{
  TargetList tl;
  QFileInfo fi(fileName());
  if(fi.exists()) {
    XMLVirtualPlugin plugin(&tl, "DispersionCurve");
    XMLDinverHeader hdr(&plugin);
    if(hdr.xml_restoreFile(fileName())!=XMLClass::NoError) {
      App::log(tr("gptarget: error reading file %1\n").arg(fileName()) );
      return 2;
    }
  }
  if(_position) {
    tl.setPosition(*_position);
  }
  switch(_type) {
  case Dispersion: {
      QList<ModalCurve> curves=tl.dispersionTarget().curves();
      tl.dispersionTarget().setMisfitWeight(_weight);
      tl.dispersionTarget().setMinimumMisfit(_minimumMisfit);
      QTextStream sIn(stdin);
      readCurves(curves, sIn);
      tl.dispersionTarget().setCurves(curves);
      XMLVirtualPlugin plugin(&tl, "DispersionCurve");
      XMLDinverHeader hdr(&plugin);
      if(hdr.xml_saveFile(fileName())!=XMLClass::NoError) {
        App::log(tr("gptarget: error writing to file %1\n").arg(fileName()) );
        return 2;
      }
    }
    break;
  case EllipticityCurve: {
      QList<ModalCurve> curves=tl.ellipticityCurveTarget().curves();
      tl.ellipticityCurveTarget().setMisfitWeight(_weight);
      tl.ellipticityCurveTarget().setMinimumMisfit(_minimumMisfit);
      QTextStream sIn(stdin);
      readCurves(curves, sIn);
      tl.ellipticityCurveTarget().setCurves(curves);
      XMLVirtualPlugin plugin(&tl, "DispersionCurve");
      XMLDinverHeader hdr(&plugin);
      if(hdr.xml_saveFile(fileName())!=XMLClass::NoError) {
        App::log(tr("gptarget: error writing to file %1\n").arg(fileName()) );
        return 2;
      }
    }
    break;
  case MagnetoTelluric: {
      QList<MagnetoTelluricCurve> curves=tl.magnetoTelluricTarget().curves();
      tl.magnetoTelluricTarget().setMisfitWeight(_weight);
      tl.magnetoTelluricTarget().setMinimumMisfit(_minimumMisfit);
      QTextStream sIn(stdin);
      readCurves(curves, sIn);
      tl.magnetoTelluricTarget().setCurves(curves);
      XMLVirtualPlugin plugin(&tl, "DispersionCurve");
      XMLDinverHeader hdr(&plugin);
      if(hdr.xml_saveFile(fileName())!=XMLClass::NoError) {
        App::log(tr("gptarget: error writing to file %1\n").arg(fileName()) );
        return 2;
      }
    }
    break;
  default:
    App::log(tr("Not yet implemented for this kind of curve\n") );
    return 2;
  }
  return 0;
}

void TargetAdd::readCurves(QList<ModalCurve>& curves, QTextStream& s)
{
  ModalCurve curve;
  curve.addMode(_mode);
  FactoryPoint p;
  QString buf;
  CoreApplication::instance()->debugUserInterrupts(false);
  while(!s.atEnd()) {
    buf=s.readLine();
    if(buf[0]=='\n' || buf[0]=='#') {
      if(!curve.isEmpty()) {
        curves.append(curve);
        curve.clear();
      }
    } else {
      p.fromString(buf);
      curve.append(p);
    }
  }
  if(!curve.isEmpty()) {
    curves.append(curve);
  }
  CoreApplication::instance()->debugUserInterrupts(true);
}

void TargetAdd::readCurves(QList<MagnetoTelluricCurve>& curves, QTextStream& s)
{
  MagnetoTelluricCurve curve;
  MagnetoTelluricPoint p;
  QString buf;
  while(!s.atEnd()) {
    buf=s.readLine();
    if(buf[0]=='\n' || buf[0]=='#') {
      if(!curve.isEmpty()) {
        curves.append(curve);
        curve.clear();
      }
    } else {
      p.fromString(buf);
      curve.append(p);
    }
  }
  if(!curve.isEmpty()) {
    curves.append(curve);
  }
}
