/***************************************************************************
**
**  This file is part of gplivemodel.
**
**  gplivemodel is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gplivemodel is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-10-16
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreWave.h>
#include <SciFigs.h>
#include "DispersionGridThread.h"
#include "DispersionThread.h"

/*!
  \class DispersionGridThread DispersionGridThread.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
DispersionGridThread::DispersionGridThread(QObject * parent)
    : Seismic1DThread(parent)
{
  TRACE;
  _polarization=Mode::Rayleigh;
  _vMinRange=100.0;
  _vMaxRange=1000.0;
}

DispersionGridThread::DispersionGridThread(const DispersionGridThread& o)
    : Seismic1DThread(o)
{
  TRACE;
  _polarization=o._polarization;
  _vMinRange=o._vMinRange;
  _vMaxRange=o._vMaxRange;
}

/*!
  Description of destructor still missing
*/
DispersionGridThread::~DispersionGridThread()
{
  TRACE;
}

void DispersionGridThread::initResultsGraphs(GraphicSheet * sheet)
{
  TRACE;
  AxisWindow * w=new AxisWindow;
  w->setGeometry(sheet->printRight(), 0.5, 10.0, 8.0);
  sheet->addObject(w);
  sheet->showObject(w);
  sheet->autoResizeContent();

  _layer=new LiveGridLayer(w);
  _layer->setObjectName(objectName()+" grid");
  _layer->setForceMinimumValue(true);
  _layer->setMinimumValue( -10.0);
  _layer->setForceMaximumValue(true);
  _layer->setMaximumValue(10.0);

  w->xAxis()->setFrequency();
  w->yAxis()->setTitle(tr("Slowness (s/m)"));
  w->yAxis()->setTitleInverseScale(tr("Velocity (m/s)"));
}

bool DispersionGridThread::setParameters(int& argc, char ** argv)
{
  TRACE;
  int i, j=1;
  for(i=1; i<argc; i++) {
    QByteArray arg=argv[i];
    if(arg[0]=='-') {
      if(arg=="-grid") {
        CoreApplication::checkOptionArg(i, argc, argv);
        if(argv[i][0]=='R') {
          _polarization=Mode::Rayleigh;
        } else {
          _polarization=Mode::Love;
        }
      } else if(arg=="-vmin") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _vMinRange=CoreApplication::toDouble(i, i-1, argv);
      } else if(arg=="-vmax") {
        CoreApplication::checkOptionArg(i, argc, argv);
        _vMaxRange=CoreApplication::toDouble(i, i-1, argv);
      } else {
        argv[j++]=argv[i];
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j < argc) {
    argv[j]=nullptr;
    argc=j;
  }
  return true;
}

void DispersionGridThread::start(QList<GeophysicalModel *> models)
{
  TRACE;
  ASSERT(models.count()==1);
  delete _layer->takeFunction();
  AbstractFunction * f;
  if(_polarization==Mode::Rayleigh) {
    RayleighFunction * rf=new RayleighFunction;
    rf->setModel(static_cast<Seismic1DModel *>(models.first()));
    f=rf;
  } else {
    LoveFunction * lf=new LoveFunction;
    lf->setModel(static_cast<Seismic1DModel *>(models.first()));
    f=lf;
  }
  _layer->setFunction(new FunctionPlot(2, f));
  resultsDeepUpdate();
}

void DispersionGridThread::setLimits()
{
  TRACE;
  _layer->graph()->xAxis()->setRange(_xMinRange, _xMaxRange);
  _layer->graph()->yAxis()->setRange(_vMinRange*0.9, _vMaxRange*1.1);
}

void DispersionGridThread::resultsDeepUpdate()
{
  TRACE;
  _layer->graph()->deepUpdate();
}
