/***************************************************************************
**
**  This file is part of gpfksimulator.
**
**  gpfksimulator is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpfksimulator is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-03-27
**  Copyright: 2006-2019
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <ArrayCore.h>

#include "Simulator.h"
#include "MediumParameters.h"
#include "SourceParameters.h"
#include "SourceItemModel.h"
#include "SourceDelegate.h"
#include "SimulatedArrayStations.h"

Simulator::Simulator(QWidget *parent, Qt::WindowFlags f)
    : QWidget(parent, f | Qt::Window)
{
  TRACE;

  setupUi(this);

  SourceItemModel * model=new SourceItemModel(&_sources, this);
  connect(model, SIGNAL(dataChanged(QModelIndex,QModelIndex)), this, SLOT(setSource(QModelIndex,QModelIndex)));
  sourceTable->setModel(model);
  SourceDelegate * delegate=new SourceDelegate(this);
  connect(sourceTable, SIGNAL(clicked(const QModelIndex&)), delegate, SLOT(colorClicked(QModelIndex)));
  sourceTable->setItemDelegate(delegate);
  sourceTable->setSelectionBehavior(QAbstractItemView::SelectItems);
  sourceTable->setSelectionMode(QAbstractItemView::ExtendedSelection);
  sourceTable->setEditTriggers(QAbstractItemView::AllEditTriggers);

  connect(planeWavesButton, SIGNAL(toggled(bool)), this, SLOT(setWaveModel()));
  connect(circularWavesButton, SIGNAL(toggled(bool)), this, SLOT(setWaveModel()));
  connect(saveSources, SIGNAL(clicked(bool)), this, SLOT(save()));
  connect(loadSources, SIGNAL(clicked(bool)), this, SLOT(load()));

  _gridLayer=nullptr;
  _maxLayer=nullptr;
  _velocityLayer=nullptr;
  _azimuthLayer=nullptr;
  _kLayer=nullptr;
  _fkMeshLayer=nullptr;
  _array=nullptr;
  _arraySelection=nullptr;
  _crossSpectrum=nullptr;
  _cache=nullptr;
  _kmaxSolver=nullptr;
  _crossSection=false;
}

Simulator::~Simulator()
{
  TRACE;
  // Force termination of painting events if any.
  _gridLayer->lockDelayPainting();
  _gridLayer->unlock();
  delete _crossSpectrum;
  delete _cache;
  delete _arraySelection; // array is used by cache in its destructor
  delete _array;
}

void Simulator::resizeEvent (QResizeEvent * )
{
  TRACE;
}

void Simulator::init(const QList<NamedPoint>& array, double sampling)
{
  TRACE;
  _array=new SimulatedArrayStations(array);
  _array->setReference();
  _array->setRelativePos(_parameters.magneticDeclination());
  _arraySelection=new ArraySelection(_array);
  _crossSpectrum=new SimulatedCrossSpectrum(_arraySelection, &_parameters);
  _cache=new FKCache(_arraySelection);

  processType->blockSignals(true);
  processType->addItems(FKParameters::processTypes());
  processType->blockSignals(false);

  // Wavenumber map
  waveNumMap->xAxis()->setTitle(tr("Wave number X (rad/m)"));
  waveNumMap->xAxis()->setTitleInverseScale(tr("Wave length X/(2*pi) (m/rad)"));
  waveNumMap->yAxis()->setTitle(tr("Wave number Y (rad/m)"));
  waveNumMap->yAxis()->setTitleInverseScale(tr("Wave length Y/(2*pi) (m/rad)"));

  _velocityLayer=new CircleViewer(waveNumMap);
  _velocityLayer->setObjectName("velocity");
  _azimuthLayer=new LineLayer(waveNumMap);
  _azimuthLayer->setObjectName("azimuth");
  PlotLine2D * line=new PlotLine2D;
  line->setPen(Pen(Qt::black, 0.6, Pen::SolidLine));
  line->setSymbol(Symbol());
  _azimuthLayer->setReferenceLine(line);

  _gridLayer=new LiveGridLayer(waveNumMap);
  _sampling=sampling;
  if(sampling<1.0) {
    sampling=1.0;
  }
  _gridLayer->setSampling(qRound(sampling));
  _gridLayer->setFunction(new FKFunctionPlot);
  _gridLayer->setOpacity(0.8);
  _gridLayer->setObjectName("wave number map");
  ColorMap& map=_gridLayer->beginColorMapChange();
  map.generate(0, 19, ColorPalette::defaultSequential, ColorPalette::Reversed);
  map.setValues(0.0, 1.0, SamplingParameters::Linear);
  _gridLayer->endColorMapChange();

  _maxLayer=new NameLineLayer(waveNumMap);
  _maxLayer->setObjectName("max");
  NameLine * lineMax=new NameLine;
  lineMax->setPen(Pen(Qt::NoPen));
  lineMax->setSymbol(Symbol(Symbol::Circle, 4, Pen(Qt::black, 0.5)));
  _maxLayer->setReferenceLine(lineMax);
  _maxLayer->addLine();

  _kLayer=new CircleMask(waveNumMap);
  _kLayer->setObjectName("grid size");
  _kLayer->setOpacity(0.75);

  _fkMeshLayer=new MeshLayer(waveNumMap);
  _fkMeshLayer->setObjectName("fk mesh");
  _fkMeshLayer->setOpacity(0.05);

  waveNumMap->addFreePicks();

  // Estimate kmax from station coordinates
  KminSolver kminSolver(_array->relativePos());
  bool ok=true;
  _parameters.setKmin(kminSolver.calculate(ok));
  if(ok && _parameters.kmin()>0.0) {
    _kmaxSolver=new KmaxSolver(_array->relativePos(), _parameters.kmin(), 0.25);
    connect(&_kmaxTimer, SIGNAL(timeout()), this, SLOT(setKmax()));
    connect(_kmaxSolver, SIGNAL(finished()), this, SLOT(setKmax()));
    _kmaxSolver->calculate();
    _kmaxTimer.start(1000);
  } else {
    _kmaxSolver=nullptr;
    App::log(tr("Error computing kmin for station coordinates\n") );
    setKmax(5.0*theoreticalKmax());
  }
}

void Simulator::initUserValues()
{
  if(_kmaxSolver) {
    App::log(tr("Waiting for Kmax computation...\n") );
    _kmaxSolver->wait();
    Application::processEvents(QEventLoop::ExcludeUserInputEvents);
  }
  setWaveModel();
  // on_frequencyEdit_valueChanged() not called to avoid calling setAllSources()
  // before a complete initialization.
  _fkMeshLayer->setFrequency(frequencyEdit->value());
  _medium.setFrequency(frequencyEdit->value());
  _medium.setAttenuation(attenuationEdit->value());
  _parameters.setVerticalNoise(sqrt(verticalNoiseEdit->value()));
  _parameters.setHorizontalNoise(sqrt(horizontalNoiseEdit->value()));
  on_gridSizeEdit_valueChanged(gridSizeEdit->value());
  on_addSource_clicked();
  setAllSources();
  on_processType_currentIndexChanged(processType->currentIndex());
  on_plotTypeEdit_currentIndexChanged(plotTypeEdit->currentIndex());
}

double Simulator::theoreticalKmax() const
{
  KminSolver s(_array->relativePos());
  double dmin, dmax;
  s.distanceRange(dmin, dmax);
  // k=2*M_PI/lambda, due to Nyquist, d<lambda/2
  return M_PI/dmin;
}

void Simulator::setKmax()
{
  TRACE;
  if(_kmaxSolver) {
    bool ok;
    double kmax=_kmaxSolver->kmax(ok);
    if(ok || !_kmaxSolver->isRunning()) {
      _kmaxTimer.stop();
      _kmaxSolver->deleteLater();
      _kmaxSolver=nullptr;
    }
    setKmax(kmax);
  }
}

void Simulator::setKmax(double kmax)
{
  TRACE;
  _parameters.setKmax(kmax);
  gridSizeEdit->setValue(_parameters.effectiveGridSize(frequencyEdit->value()));
}

void Simulator::setWaveModel()
{
  TRACE;
  if(planeWavesButton->isChecked()) {
    attenuationEdit->setEnabled(false);
    attenuationButton->setEnabled(false);
    static_cast<SourceItemModel *>(sourceTable->model())->setWaveModel(SourceParameters::PlaneWaves);
    static_cast<SourceDelegate *>(sourceTable->itemDelegate())->setWaveModel(SourceParameters::PlaneWaves);
    _crossSpectrum->setWaveModel(SourceParameters::PlaneWaves);
  } else {
    attenuationButton->setEnabled(true);
    attenuationEdit->setEnabled(attenuationButton->isChecked());
    static_cast<SourceItemModel *>(sourceTable->model())->setWaveModel(SourceParameters::CircularWaves);
    static_cast<SourceDelegate *>(sourceTable->itemDelegate())->setWaveModel(SourceParameters::CircularWaves);
    _crossSpectrum->setWaveModel(SourceParameters::CircularWaves);
  }
  setAllSources();
}

void Simulator::on_randomBut_clicked()
{
  setAllSources();
}

void Simulator::on_frequencyEdit_valueChanged(double f)
{
  TRACE;
  _medium.setFrequency(f);
  gridSizeEdit->setValue(_parameters.effectiveGridSize(f));
  _fkMeshLayer->setFrequency(f);
  setAllSources();
}

void Simulator::on_processType_currentIndexChanged(int index)
{
  TRACE;
  FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
  crossSectionBut->setEnabled(false);
  // +1 to skip Úndefined'process type.
  FKParameters::ProcessType t=static_cast<FKParameters::ProcessType>(index+1);
  _parameters.setProcessType(t);
  AbstractFKFunction * fk=AbstractFKFunction::create(_array->mode(), &_parameters, _cache);
  if(fk) {
    _crossSpectrum->resize();
    fk->setCrossSpectrum(_crossSpectrum);
    fk->initialize(_parameters);
    bool ok=_crossSpectrum->calculate(blockCountEdit->value(), fk, _parameters);
     _gridLayer->setFunctionValid(ok);
     funcPlot->setFunction(fk);
  } else {
    _gridLayer->setFunctionValid(false);
  }
  _gridLayer->setFunction(funcPlot);
  updateMap();
}

void Simulator::on_sensorRotationEdit_valueChanged(double a)
{
  TRACE;
  FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
  _crossSpectrum->setSensorRotation(a);
  _crossSpectrum->setStationSignals(blockCountEdit->value(),
                                    _parameters.verticalNoise(),
                                    _parameters.horizontalNoise());
  if(funcPlot->function()) {
    bool ok=_crossSpectrum->calculate(blockCountEdit->value(), funcPlot->function(), _parameters);
    _gridLayer->setFunctionValid(ok);
  } else {
    _gridLayer->setFunctionValid(false);
  }
  _gridLayer->setFunction(funcPlot);
  updateMap();
}

void Simulator::on_verticalNoiseEdit_valueChanged(double a)
{
  TRACE;
  FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
  _parameters.setVerticalNoise(sqrt(a));
  _crossSpectrum->setStationSignals(blockCountEdit->value(),
                                    _parameters.verticalNoise(),
                                    _parameters.horizontalNoise());
  if(funcPlot->function()) {
    bool ok=_crossSpectrum->calculate(blockCountEdit->value(), funcPlot->function(), _parameters);
    _gridLayer->setFunctionValid(ok);
  } else {
    _gridLayer->setFunctionValid(false);
  }
  _gridLayer->setFunction(funcPlot);
  updateMap();
}

void Simulator::on_horizontalNoiseEdit_valueChanged(double a)
{
  TRACE;
  FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
  _parameters.setHorizontalNoise(sqrt(a));
  _crossSpectrum->setStationSignals(blockCountEdit->value(),
                                    _parameters.verticalNoise(),
                                    _parameters.horizontalNoise());
  if(funcPlot->function()) {
    bool ok=_crossSpectrum->calculate(blockCountEdit->value(), funcPlot->function(), _parameters);
    _gridLayer->setFunctionValid(ok);
  } else {
    _gridLayer->setFunctionValid(false);
  }
  _gridLayer->setFunction(funcPlot);
  updateMap();
}

void Simulator::on_plotTypeEdit_currentIndexChanged(int index)
{
  TRACE;
  FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
  fixedEllLabel->setEnabled(false);
  fixedEll->setEnabled(false);
  switch(index) {
  case 0:
    funcPlot->setMode(FKFunctionPlot::BeamPower);
    break;
  case 1:
    funcPlot->setMode(FKFunctionPlot::Ellipticity);
    break;
  case 2:
    funcPlot->setMode(FKFunctionPlot::FixedEllipticity);
    funcPlot->setFixedEllipticity(Angle::degreesToRadians(fixedEll->value()));
    fixedEllLabel->setEnabled(true);
    fixedEll->setEnabled(true);
    break;
  }
  _gridLayer->setFunction(funcPlot);
  updateMap();
}

void Simulator::on_fixedEll_valueChanged(double value)
{
  TRACE;
  FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
  funcPlot->setFixedEllipticity(Angle::degreesToRadians(value));
  _gridLayer->setFunction(funcPlot);
  updateMap();
}

void Simulator::on_blockCountEdit_valueChanged(int)
{
  TRACE;
  setAllSources();
}

void Simulator::on_attenuationEdit_valueChanged(double q)
{
  TRACE;
  _medium.setAttenuation(q);
  setAllSources();
}

void Simulator::on_attenuationButton_toggled(bool checked)
{
  TRACE;
  if(checked) {
    _medium.setAttenuation(attenuationEdit->value());
    attenuationEdit->setEnabled(true);
  } else {
    _medium.setAttenuation(0.0);
    attenuationEdit->setEnabled(false);
  }
  setAllSources();
}

void Simulator::on_gridSizeEdit_valueChanged(double k)
{
  TRACE;
  _kLayer->set(0, 0, k, k, Qt::white);
  _kLayer->deepUpdate();
  waveNumMap->xAxis()->setRange(-k, k);
  waveNumMap->yAxis()->setRange(-k, k);
}

void Simulator::on_addSource_clicked()
{
  TRACE;
  static_cast<SourceItemModel *>(sourceTable->model())->addSource();
  _crossSpectrum->addSource();
  addSourceMarkers();
  setSource(_crossSpectrum->sourceCount()-1);
}

void Simulator::addSourceMarkers()
{
  TRACE;
  while(_velocityLayer->count()<_sources.count()) {
    _velocityLayer->add(0, 0.0, 0.0, 0.0, 0.0, Pen());
    AbstractLine * l=_azimuthLayer->addLine();
    l->append();
    l->append();
    l->setX(0, 0.0);
    l->setY(0, 0.0, nullptr);
  }
}

void Simulator::on_removeSource_clicked()
{
  TRACE;
  int index=sourceTable->currentIndex().row();
  if(index>=0) {
    static_cast<SourceItemModel *>(sourceTable->model())->removeSource(index);
    _crossSpectrum->removeSource(index);
    FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
    _crossSpectrum->setStationSignals(blockCountEdit->value(),
                                      _parameters.verticalNoise(),
                                      _parameters.horizontalNoise());
    bool ok=_crossSpectrum->calculate(blockCountEdit->value(), funcPlot->function(), _parameters);
    _gridLayer->setFunctionValid(ok);
    _velocityLayer->remove(index);
    _azimuthLayer->removeLine(index);
    _gridLayer->setFunction(funcPlot);
    updateMap();
  }
}

void Simulator::on_crossSectionBut_clicked()
{
  TRACE;
  QString fileName=Message::getSaveFileName(tr("Cross-section"), tr("Text file (*)"));
  if(!fileName.isEmpty()) {
    QFile f(fileName);
    if(f.open(QIODevice::WriteOnly)) {
      QTextStream s(&f);
      //double xmin=waveNumMap->xAxis()->minimum();
      //double dx=(waveNumMap->xAxis()->maximum()-xmin)/200.0;
      //PrivateVector<double> k(3, 0.0);
      TODO_WARNING;
      /*
      s << "#\n";
      for(int i=0; i<=200; i++) {
        k[0]=xmin+(double)i*dx;
        s << QString::number(k.x(), 'f', 20) << " " << QString::number(_fkmap->HRFKRayleigh(k), 'f', 20) << Qt::endl;
      }
      */
      /*
      for(double v=190; v<=210; v+=2) {
        k[0]=10.0*2.0*M_PI/v;
        s << "#\n";
        double& ell=k[2];
        for(ell=-2.5; ell<=2.5; ell+=0.01) {
          s << QString::number(k.z(), 'f', 20) << " " << QString::number(_fkmap->HRFKRayleigh(k), 'f', 20) << Qt::endl;
        }
      }
      k[0]=10.0*2.0*M_PI/200;
      double& ell=k[2];
      for(ell=-2.5; ell<=2.5; ell+=0.01) {
        s << QString::number(k[2], 'f', 20) << " " << QString::number(_fkmap->HRFKRayleigh(k), 'f', 20) << Qt::endl;
      }
      */
    }
  }
}

void Simulator::on_maximaBut_clicked()
{
  TRACE;
  FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
  AbstractFKFunction * fk=funcPlot->function();

  _parameters.setMaximumPeakCount(maximaCount->value());
  GridSearch2D * grid=static_cast<GridSearch2D *>(fk->createSearch(_parameters));
  double gridSize=_parameters.effectiveGridSize(_medium.frequency());
  if(gridSize>gridSizeEdit->value()) {
    gridSize=gridSizeEdit->value();
  }
  fk->setMaximumWavenumber(gridSize);
  fk->setMaximumSlowness(_parameters.maximumSlowness());
  fk->setGridStep(_parameters.gridStep());
  double omega=2.0*M_PI*_medium.frequency();
  fk->setTotalLimits(omega*omega);
  fk->setAzimuthRange(_parameters.minimumAzimuth(),
                      _parameters.maximumAzimuth());
  _fkMeshLayer->setStep(_parameters.gridStep());
  _fkMeshLayer->setSize(waveNumMap->xAxis()->visibleMaximum());
  Curve<NamedPoint>& plot=static_cast<NameLine *>(_maxLayer->line(0))->curve();
  plot.clear();
  if(maximaCount->value()==1) {
    grid->globalMax();
  } else {
    grid->localMax(_parameters.maximumPeakCount(_array->count()),
                   _parameters.absoluteThreshold(),
                   _parameters.relativeThreshold());
  }
  FunctionMaximaIterator it;
  double dkMin=std::numeric_limits<double>::infinity();
  double freq=_medium.frequency();
  FKPeaks::Value value;
  fprintf(stderr, "%3s %12s %12s %12s %12s %12s %12s %12s %12s %12s %12s\n",
         "N", "kx (rad/m)", "ky (rad/m)", "v (m/s)", "az. (deg)", "xi (deg)", "xi_r (deg)", "Sigma", "Rz/N", "Rh/N", "power");
  int i=0;
  for(it=grid->begin(); it!=grid->end(); ++it) {
    const FunctionSearchMaximum& m=**it;
    const Vector<double>& kell=m.position();
    if(!fk->isOnLimit(kell, _medium.frequency())) {
      double dk=distanceToSources(kell);
      if(dk<dkMin) {
        dkMin=dk;
      }
      fk->setAssociatedResults(kell, m.value(), value);
      fprintf(stderr, "%3i %12lf %12lf %12.2lf %12.2lf %12.2lf %12.2lf %12.2lf %12.2lf %12.2lf %12lg\n",
             i++, kell[0], kell[1], 2.0*M_PI*freq/kell.length(0, 1),
             Angle::radiansToDegrees(Angle::mathToGeographic(kell.azimuth(0, 1))),
             Angle::radiansToDegrees(kell[2]), value.ellipticity,
             value.horizontalNoise/value.verticalNoise,
             value.verticalNoise, value.horizontalNoise, value.power);
      NamedPoint p(kell[0], kell[1], 0.0, QString::number((*it)->value(), 'g', 2));
      plot.append(p);
    }
  }
  fprintf(stderr, "Minimum distance to a source on the FK map: %lf rad/m \n", dkMin);
  //fitCrossSpectrum(grid);
  grid->takeFunction();
  delete grid;
  // back to plot all peaks even greater than kmax
  fk->setMaximumWavenumber(std::numeric_limits<double>::infinity());
  _gridLayer->setFunction(funcPlot);
  _maxLayer->deepUpdate();
}

void Simulator::fitCrossSpectrum(GridSearch2D * grid)
{
  if(processType->currentText()!="RDSCross") {
    return;
  }
  AbstractFKRayleigh * fk=static_cast<AbstractFKRayleigh *>(grid->function());
  WavefieldFunction * wf=new WavefieldFunction(blockCountEdit->value(),
                                               _arraySelection->relativePos());
  FunctionMaximaIterator itg=grid->begin();
  IncoherentNoise noise;
  FKPeaks::Value value;
  int stationCount=_crossSpectrum->array().count();
  for(int i=0; i<grid->maximaCount(); i++, ++itg) {
    const FunctionSearchMaximum& m=**itg;
    const Vector<double>& kell=m.position();
    printf("Wave %i: %lf %lf\n", i, 2.0*M_PI*_medium.frequency()/kell.length(0, 1),
                                 Angle::radiansToDegrees(Angle::mathToGeographic(kell.azimuth(0, 1))));
    char buf[8];
    printf("Include this wave ? [Yn] ");
    scanf("%s", buf);
    if(strcmp(buf,"n")==0) {
      continue;
    }
    printf("\n");

    fk->setAssociatedResults(kell, m.value(), value);
    noise.add(value.power, value.horizontalNoise*stationCount, value.verticalNoise*stationCount);

    WavefieldValues::Observations obs;
    obs.xiX=kell[2];
    obs.powerX=fk->valueFixed(kell);
    obs.kConcavityX=fk->concavity(kell, 0);
    obs.xiConcavityX=fk->concavity(kell, 2);
    printf("     X %lf %lf %lf %lf\n", obs.powerX, Angle::radiansToDegrees(obs.xiX),
                                 obs.kConcavityX, obs.xiConcavityX);
    PrivateVector<double> kellh(kell);
    obs.powerH=fk->radialPower(kellh);
    obs.xiH=kellh[2];
    obs.kConcavityH=fk->radialConcavity(kellh, 0);
    obs.xiConcavityH=fk->radialConcavity(kellh, 2);
    printf("     H %lf %lf %lf %lf\n", obs.powerH, Angle::radiansToDegrees(obs.xiH),
                                 obs.kConcavityH, obs.xiConcavityH);
    PrivateVector<double> kellz(kell);
    obs.powerZ=fk->verticalPower(kellz);
    obs.xiZ=kellz[2];
    obs.kConcavityZ=fk->verticalConcavity(kellz, 0);
    obs.xiConcavityZ=fk->verticalConcavity(kellz, 2);
    printf("     Z %lf %lf %lf %lf\n", obs.powerZ, Angle::radiansToDegrees(obs.xiZ),
                                 obs.kConcavityZ, obs.xiConcavityZ);
    wf->addObservedWave(kell.length(0, 1), kell.azimuth(0, 1), obs);
  }
  wf->setSteering(static_cast<HRFKDirectRayleighAll *>(fk)->steering());
  wf->wavefield().setSignalPower();
  wf->saveObservations("/home/wathelem/publis/3c_corr/M2.1/RTBF/observations");

  OptimizationBFGS opt(wf->parameterSpaceDimension());
  opt.setFunction(wf);
  PrivateVector<double> maxDp(wf->parameterSpaceDimension());
  PrivateVector<double> p(wf->parameterSpaceDimension());
  PrivateVector<double> prec(wf->parameterSpaceDimension());
  wf->setMaximumShift(maxDp);
  wf->setPrecision(prec);
  opt.setPrecision(prec);

  noise.showStatus();
  wf->wavefield().setIncoherentNoise(noise.R(), noise.sigma());
  wf->wavefield().getParameters(p);
  opt.reset(p);
  wf->showDetailedMisfit();
  opt.minimize(maxDp, 100);
  wf->showDetailedMisfit();
  wf->showResults(opt.value(), opt.iterationCount(), _medium.frequency());

  /*SamplingParameters samp;
  samp.setRange(0.2, 5.0);
  samp.setScaleType(SamplingParameters::Log);
  samp.setStepType(SamplingParameters::Step);
  samp.setStep(1.2);


  for(int i=0; i<samp.count(); i++) {
    p0[2]=samp.value(i);
    for(int j=0; j<samp.count(); j++) {
      p0[3]=samp.value(j);
      App::log(tr("Init at R %1 sigma %2\n").arg(p0[2]).arg(p0[3]));
      opt.reset(p0);
      wf->showDetailedMisfit();
      opt.minimize(maxDp, 100);
      wf->showDetailedMisfit();
      wf->showResults(opt.value(), opt.iterationCount());
    }
  }*/

  /*Wavefield wp;
  wp.setBlockCount(blockCountEdit->value());
  wp.setSensors(_arraySelection->relativePos());
  wp.setWaveCount(grid->maximaCount());

  FunctionMaximaIterator it=grid->begin();
  for(int i=0; i<grid->maximaCount(); i++, ++it) {
    const FunctionSearchMaximum& m=**it;
    const Vector<double>& kell=m.position();
    wp.setWave(i, 1.0, kell.length(0, 1), kell.azimuth(0, 1), kell[2]);
    printf("Wave %i: %lf %lf %lf\n", i, kell.length(0, 1),
                                        Angle::radiansToDegrees(kell.azimuth(0, 1)),
                                        Angle::radiansToDegrees(kell[2]));
  }
  wp.setIncoherentNoise(1.5, 0.75);
  wp.setFieldVectors();

  // Show simulated FK map
  ComplexMatrix th=wp.crossSpectrum();
  th.invert();*/

  /*fk->setCrossSpectrum(&th);
  QFile ffk("/tmp/fksim");
  ffk.open(QIODevice::WriteOnly);
  QTextStream sfk(&ffk);
  sfk << "x y val\n";
  for(double kx=-1; kx<1; kx+=0.01) {
    for(double ky=-1; ky<1; ky+=0.01) {
      PrivateVector<double> k(2);
      k.setValues(kx, ky);
      sfk << kx << " " << ky << " " << fk->value(k) << "\n";
    }
  }
  ffk.close();*/

  // Print peak values for obs and sim
  /*it=grid->begin();
  for(int i=0; i<grid->maximaCount(); i++, ++it) {
    const FunctionSearchMaximum& m=**it;
    const Vector<double>& kell=m.position();
    fk->setCrossSpectrum(_crossSpectrum->matrix());
    double p, ddp;
    p=fk->valueFixed(kell);
    ddp=fk->concavity(kell);
    printf("        obs beampower %lf ddp %lf\n", p, ddp);
    fk->setCrossSpectrum(&th);
    p=fk->valueFixed(kell);
    ddp=fk->concavity(kell);
    printf("        sim beampower %lf ddp %lf\n", p, ddp);
  }*/

#if 0
  // Parameter sensitivity
  App::freeze(true);
  for(int i=0; i<grid->maximaCount(); i++) {
    //const FunctionSearchMaximum& m=**it;
    //const Vector<double>& kell=m.position();
    //fk->setCrossSpectrum(_crossSpectrum->matrix());
    //double pobs=fk->valueFixed(kell);
    //double ddpobs=fk->concavity(kell);

    QFile f("/tmp/grid"+QString::number(i));
    f.open(QIODevice::WriteOnly);
    QTextStream s(&f);
    s << "x y val\n";
    int nd=wf->parameterSpaceDimension();
    PrivateVector<double> param(nd);
    param.copyValues(p0);
    /*for(double xi1=-M_PI/2; xi1<M_PI/2; xi1+=M_PI/180.0) {
      param[2*i+1]=xi1;
      for(double a=0.2; a<2; a+=0.05) {
        param[2*i]=a;
        s << Angle::radiansToDegrees(xi1) << " " << a << " " << -wf->value(param) << "\n";
      }
    }*/
    for(double etah=0.2; etah<5; etah+=0.05) {
      param[2]=etah;
      for(double etaz=0.2; etaz<5; etaz+=0.05) {
        param[3]=etaz;
        s << etah << " " << etaz << " " << -wf->value(param) << "\n";
      }
    }
    wf->wavefield().setParameters(p0);
    f.close();
  }
  App::freeze(false);
#endif

  // Revert to original cross-spectrum matrix
  fk->setCrossSpectrum(_crossSpectrum->matrix());
}

void Simulator::displaySource(int index)
{
  TRACE;
  const SourceParameters& src=_sources.at(index);
  QColor col;
  guiColor(src.color(), col);
  double v=src.velocity();
  double k=2.0*M_PI*_medium.frequency()/v;
  double a=src.azimuth(_crossSpectrum->waveModel());
  _velocityLayer->set(index, 0, 0, k, k, 0.0, col);
  AbstractLine * l=_azimuthLayer->line(index);
  l->setPen(Pen(col, 0.6, Pen::SolidLine));
  l->setX(1, k*cos(a));
  l->setY(1, k*sin(a), nullptr);
}

void Simulator::setSource(QModelIndex topLeft, QModelIndex)
{
  TRACE;
  setSource(topLeft.row());
  displaySource(topLeft.row());
}

void Simulator::setSource(int iSrc)
{
  TRACE;
  FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
  _crossSpectrum->setSourceSignals(iSrc, _sources.at(iSrc), _medium, blockCountEdit->value());
  _crossSpectrum->setStationSignals(blockCountEdit->value(),
                                    _parameters.verticalNoise(),
                                    _parameters.horizontalNoise());
  if(funcPlot->function()) {
    bool ok=_crossSpectrum->calculate(blockCountEdit->value(), funcPlot->function(), _parameters);
    _gridLayer->setFunctionValid(ok);
  }
  _gridLayer->setFunction(funcPlot);
  updateMap();
}

void Simulator::setAllSources()
{
  TRACE;
  FKFunctionPlot * funcPlot=static_cast<FKFunctionPlot *>(_gridLayer->takeFunction());
  int n=_crossSpectrum->sourceCount();
  for(int i=0; i<n; i++) {
    _crossSpectrum->setSourceSignals(i, _sources.at(i), _medium, blockCountEdit->value());
    displaySource(i);
  }
  _crossSpectrum->setStationSignals(blockCountEdit->value(),
                                    _parameters.verticalNoise(),
                                    _parameters.horizontalNoise());
  if(funcPlot->function()) {
    bool ok=_crossSpectrum->calculate(blockCountEdit->value(), funcPlot->function(), _parameters);
    _gridLayer->setFunctionValid(ok);
  }
  _gridLayer->setFunction(funcPlot);
  updateMap();
}

void Simulator::updateMap()
{
  TRACE;
  //if(_fkmap->polarization()==FKArrayMap::RayleighEllSign) {
  //  ellipticitySign();
  //} else {
    updateMax();
  //}
  waveNumMap->graphContents()->deepUpdate();
}

/*!
  Look for the highest peak in the current map area
*/
void Simulator::updateMax()
{
  TRACE;
  if(_parameters.kmin()>0.0 && _parameters.kmax()>0.0) {
    Curve<NamedPoint>& plot=static_cast<NameLine *>(_maxLayer->line(0))->curve();
    plot.clear();
  }
}

/*!
  Look for ellipticity sign at the peaks
*/
void Simulator::ellipticitySign()
{
  TRACE;
//  if(_kmin>0.0 && _kmax>0.0) {
//    GridSearch grid;
//    grid.setFunction(_fkmap);
//    _fkmap->setPolarization(FKArrayMap::Rayleigh);
//    _fkmap->setKmax(_kmax);
//    grid.setGrid(waveNumMap->xAxis()->visibleMinimum(),
//                 waveNumMap->xAxis()->visibleMaximum(),
//                 0.25*_kmin,
//                 waveNumMap->yAxis()->visibleMinimum(),
//                 waveNumMap->yAxis()->visibleMaximum(),
//                 0.25*_kmin);
//    Point2D k;
//    if(_globalMax) {
//      grid.globalMax();
//      k=grid.pos();
//      _fkmap->setPolarization(FKArrayMap::RayleighEllSign);
//      printf(" %lf %lf %i\n", k.x(), k.y(), _fkmap->value(k)==1.0 ? 1 : -1);
//    } else {
//      grid.localMax(INT_MAX, 0.0, 90);
//      FunctionMaximaIterator it;
//      int globalPol=-2, pol;
//      for(it=grid.begin(); it!=grid.end(); ++it) {
//        k=(*it)->pos();
//        // Check that wavenumber corresponds to a source
//        double dk=distanceToSources(k);
//        if(dk<_kmin) {
//          _fkmap->setPolarization(FKArrayMap::RayleighEllSign);
//          pol=_fkmap->value(k)==1.0 ? 1 : -1;
//          _fkmap->setPolarization(FKArrayMap::Rayleigh);
//          if(globalPol!=pol) {
//            if(globalPol==-2) {
//              globalPol=pol;
//            } else {
//              globalPol=0;
//            }
//          }
//        }
//        printf(" %i\n", globalPol);
//      }
//    }
//    grid.takeFunction();
//    _fkmap->setPolarization(FKArrayMap::RayleighEllSign);
//    _fkmap->setKmax(0.0); // back to plot all peak even > kmax
//  }
}

double Simulator::distanceToSources(const Vector<double>& k)
{
  TRACE;
  PrivateVector<double> kvs(k);
  int n=_sources.count();
  double dkMin=std::numeric_limits<double>::infinity();
  for(int i=0; i<n; i++) {
    const SourceParameters& src=_sources.at(i);
    if(src.polarization()!=Mode::Transverse) {
      double v=src.velocity();
      double ks=2.0*M_PI*_medium.frequency()/v;
      double a=src.azimuth(_crossSpectrum->waveModel());
      kvs[0]=ks*cos(a);
      kvs[1]=ks*sin(a);
      double dk=kvs.distanceTo(k);
      if(dk<dkMin) {
        dkMin=dk;
      }
    }
  }
  return dkMin;
}

bool Simulator::save(QString fileName)
{
  TRACE;
  static const QString title=tr("Save source parameters");
  if(fileName.isEmpty()) {
    fileName=Message::getSaveFileName(title, tr("Source parameters (*.src)"));
  }
  if(!fileName.isEmpty()) {
    XMLHeader hdr(&_sources);
    XMLClass::Error err=hdr.xml_saveFile(fileName, nullptr);
    if(err==XMLClass::NoError) {
      return true;
    } else {
      Message::warning(MSG_ID, title, XMLClass::message(err, fileName));
    }
  }
  return false;
}

bool Simulator::load(QString fileName)
{
  TRACE;
  static const QString title=tr("Open an existing source list");
  if(fileName.isEmpty()) {
    fileName=Message::getOpenFileName(title, tr("Source parameters (*.src)"));
  }
  if(!fileName.isEmpty()) {
    SourceList list;
    XMLHeader hdr(&list);
    XMLClass::Error err=hdr.xml_restoreFile(fileName);
    if(err==XMLClass::NoError) {
      static_cast<SourceItemModel *>(sourceTable->model())->setSources(list);
      _crossSpectrum->setSourceCount(_sources.count());
      _velocityLayer->clear();
      _azimuthLayer->clear();
      addSourceMarkers();
      setAllSources();
      return true;
    } else {
      Message::warning(MSG_ID, title, XMLClass::message(err, fileName));
    }
  }
  return false;
}

void Simulator::scan(const QStringList& sources)
{
  TRACE;
  if(_kmaxSolver) {
    _kmaxSolver->wait();
    CoreApplication::processEvents();
  }
  _gridLayer->lockDelayPainting();

  // Get the list of source to scan
  SourceList refSources;
  QList<int> sourceIndexes;
  int nSrc=_sources.count();
  for(int iSrc=0; iSrc<nSrc; iSrc++) {
    SourceParameters& src=_sources[iSrc];
    if(sources.contains(src.name())) {
      refSources.append(src);
      sourceIndexes.append(iSrc);
    }
  }
  nSrc=sourceIndexes.count();

  for(double az=0.0; az<2.0*M_PI; az+=M_PI/36.0) {
    for(double ph=0.0; ph<360; ph+=10) {
      for(int iSrc=0; iSrc<nSrc; iSrc++) {
        SourceParameters& src=_sources[sourceIndexes.at(iSrc)];
        src=refSources.at(iSrc);
        src.setAzimuth(_crossSpectrum->waveModel(), src.azimuth(_crossSpectrum->waveModel())+az);
        src.setPhase(src.phase()+ph);
        _crossSpectrum->setSourceSignals(iSrc, src, _medium, blockCountEdit->value());
      }
      _crossSpectrum->setStationSignals(blockCountEdit->value(),
                                        _parameters.verticalNoise(),
                                        _parameters.horizontalNoise());
      printf("%lf %lf ", az, ph);
      //if(_fkmap->polarization()==FKArrayMap::FK_EllipticitySign) {
      //  ellipticitySign();
      //} else {
        updateMax();
      //}
    }
  }
  _gridLayer->unlock();
}

/*!
  Computation of sigma requires |B|^2 to vanish but it is
  not necessarily true. This funtion test the computation of sigma
  on a regular grid.

  The combination of sigma and xi0 controls if the average of the factor is
  higher or lower than 1.
*/
void Simulator::averageSigma()
{
  double RN=1;
  double sigma=_parameters.horizontalNoise()/_parameters.verticalNoise();
  sigma*=sigma;
  double xi0=_sources.first().ellipticity();
  double v0=_sources.first().velocity();
  double theta0=_sources.first().azimuth(SourceParameters::PlaneWaves);
  double omega=2.0*M_PI*_medium.frequency();

  double kx0=omega/v0*cos(theta0);
  double ky0=omega/v0*sin(theta0);

  double kmax=gridSizeEdit->value();
  double kmin=kmax/100.0;

  double cxi0=cos(xi0);
  double sxi0=sin(xi0);
  double RNh=sigma/(2*sigma+1)*RN;
  double lambda=sigma*cxi0*cxi0+sxi0*sxi0+RNh;
  TheoreticalFK b2Engine(_array->relativePos());
  PrivateVector<double> dk(2, 0.0);
  Statistics statAll, statSelective;
  QFile f("/tmp/bgrid");
  f.open(QIODevice::WriteOnly);
  QTextStream s(&f);
  s << "x y val\n";
  for(double kx=-kmax; kx<=kmax; kx+=kmin) {
    for(double ky=-kmax; ky<=kmax; ky+=kmin) {
      dk[0]=kx-kx0;
      dk[1]=ky-ky0;
      double b2=b2Engine.value(dk);
      double val=(lambda-sigma*cxi0*cxi0*b2)/(lambda-sxi0*sxi0*b2);
      s << kx << " " << ky << " " << val << Qt::endl;
      statAll.add(val);
      if(b2<0.025) {
        statSelective.add(val);
      }
    }
  }
  printf("statAll %i %lf %lf statSelective %i %lf %lf\n",
         statAll.count(), statAll.mean(), statAll.stddev(),
         statSelective.count(), statSelective.mean(), statSelective.stddev());
}
