/***************************************************************************
**
**  This file is part of gpfksimulator.
**
**  gpfksimulator is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpfksimulator is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-11-08
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SimulatedCrossSpectrum.h"

/*!
  \class SimulatedCrossSpectrum SimulatedCrossSpectrum.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

GlobalRandom * SimulatedCrossSpectrum::_noisePhaseGenerator=nullptr;

/*!
  Description of constructor still missing
*/
SimulatedCrossSpectrum::SimulatedCrossSpectrum(const ArraySelection * array,
                                               const Parameters * param)
  : FKCrossSpectrum(array, param)
{
  TRACE;
  _sensorRotation=0.0;
  _stationSig=nullptr;
  _currentBlock=0;
  if(!_noisePhaseGenerator) {
    _noisePhaseGenerator=new GlobalRandom;
    CoreApplication::instance()->addGlobalObject(_noisePhaseGenerator);
  }

}

/*!
  Description of destructor still missing
*/
SimulatedCrossSpectrum::~SimulatedCrossSpectrum()
{
  TRACE;
  clearSignals();
  qDeleteAll(_sourceSig);
}

Point SimulatedCrossSpectrum::setActiveSource(const VectorList<int>&, bool& ok)
{
  if(!_sourceSig.isEmpty()) {
    Point src=_sourceSig.first()->parameters().sourcePoint();
    for(int i=_sourceSig.count()-1; i>=0; i--) {
      if(src!=_sourceSig.at(i)->parameters().sourcePoint()) {
        App::log(tr("Several source points: only the first position is considered.\n"));
        break;
      }
    }
    src=_array.array()->toAbsolute(src);
    ok=selectStations(src);
    return src;
  } else {
    ok=false;
    return Point();
  }
}

bool SimulatedCrossSpectrum::calculate(int blockCount, AbstractFKFunction * function,
                                       const Parameters& param)
{
  if(_waveModel==SourceParameters::CircularWaves) {
    if(_sourceSig.isEmpty()) {
      return false;
    }
    VectorList<int> dummy;
    if(!function->setSource(dummy)) {
      return false;
    }
  } else {
    switch(param.processType()) {
    case FKParameters::Undefined:
    FKPARAMETERS_PASSIVE_PROCESS_TYPES
      break;
    FKPARAMETERS_ACTIVE_PROCESS_TYPES
      App::log(tr("Active process types are available only for circular waves.\n"));
      return false;
    }
  }
  function->resetCrossSpectrum();
  for(_currentBlock=0; _currentBlock<blockCount; _currentBlock++) {
    function->addCrossSpectrum();
  }
  function->meanCrossSpectrum(blockCount);
  function->addIncoherentNoise(param.horizontalNoise()*param.horizontalNoise(),
                               param.verticalNoise()*param.verticalNoise());
  if(!function->invertCrossSpectrum()) {
    return false;
  }
  return true;
}

void SimulatedCrossSpectrum::addVertical()
{
  int stationCount=_array.count();
  ComplexMatrix& covmat=*_matrix;
  ComplexMatrix sig(stationCount, 1);

  for(int is=0; is<stationCount; is++) {
    sig.at(is, 0)=_stationSig[2][is][_currentBlock];
  }
  covmat+=sig*sig.conjugate().transposed();
}

void SimulatedCrossSpectrum::addRotatedRadial(int rotationIndex, const Complex * sensorRotations)
{
  int stationCount=_array.count();
  ComplexMatrix& covmat=_rotatedMatrices[rotationIndex];
  ComplexMatrix sig(stationCount, 1);
  Angle rotation;
  rotation.setRadians(_rotateStep*rotationIndex);
  Complex rsv;
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(is, 0);
    const Complex& rot=sensorRotations[is];
    rsv=_stationSig[0][is][_currentBlock]*rot.re();
    rsv-=_stationSig[1][is][_currentBlock]*rot.im();
    s=rotation.cos()*rsv;   // Rotated east
    rsv=_stationSig[1][is][_currentBlock]*rot.re();
    rsv+=_stationSig[0][is][_currentBlock]*rot.im();
    s+=rotation.sin()*rsv;  // Rotated north
  }
  covmat+=sig*sig.conjugate().transposed();
}

void SimulatedCrossSpectrum::addRotatedTransverse(int rotationIndex, const Complex * sensorRotations)
{
  int stationCount=_array.count();
  ComplexMatrix& covmat=_rotatedMatrices[rotationIndex];
  ComplexMatrix sig(stationCount, 1);
  Angle rotation;
  rotation.setRadians(_rotateStep*rotationIndex);
  Complex rsv;
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(is, 0);
    const Complex& rot=sensorRotations[is];
    rsv=_stationSig[1][is][_currentBlock]*rot.re();
    rsv+=_stationSig[0][is][_currentBlock]*rot.im();
    s=rotation.cos()*rsv;   // Rotated north
    rsv=_stationSig[0][is][_currentBlock]*rot.re();
    rsv-=_stationSig[1][is][_currentBlock]*rot.im();
    s-=rotation.sin()*rsv;  // Rotated east
  }
  covmat+=sig*sig.conjugate().transposed();
}

void SimulatedCrossSpectrum::addRotatedRayleigh(int rotationIndex, const Complex * sensorRotations)
{
  int stationCount=_array.count();
  int stationCount2=2*stationCount;
  ComplexMatrix& covmat=_rotatedMatrices[rotationIndex];
  ComplexMatrix sig(stationCount2, 1);
  Angle rotation;
  rotation.setRadians(_rotateStep*rotationIndex);
  Complex rsv;
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(is, 0);
    const Complex& rot=sensorRotations[is];
    rsv=_stationSig[0][is][_currentBlock]*rot.re();
    rsv-=_stationSig[1][is][_currentBlock]*rot.im();
    s=rotation.cos()*rsv;   // Rotated east
    rsv=_stationSig[1][is][_currentBlock]*rot.re();
    rsv+=_stationSig[0][is][_currentBlock]*rot.im();
    s+=rotation.sin()*rsv;  // Rotated north
  }
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(stationCount+is, 0);
    s=_stationSig[2][is][_currentBlock];
  }
  covmat+=sig*sig.conjugate().transposed();
}

void SimulatedCrossSpectrum::addThreeComponent(const Complex * sensorRotations)
{
  TRACE;
  int stationCount=_array.count();
  int stationCount2=2*stationCount;
  ComplexMatrix sig(3*stationCount, 1);
  ComplexMatrix& covmat=*_matrix;
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(is, 0);
    const Complex& rot=sensorRotations[is];
    s=_stationSig[0][is][_currentBlock]*rot.re();
    s-=_stationSig[1][is][_currentBlock]*rot.im();
  }
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(is+stationCount, 0);
    const Complex& rot=sensorRotations[is];
    s=_stationSig[1][is][_currentBlock]*rot.re();
    s+=_stationSig[0][is][_currentBlock]*rot.im();
  }
  for(int iStat=0; iStat<stationCount; iStat++) {
    Complex& s=sig.at(iStat+stationCount2, 0);
    s=_stationSig[2][iStat][_currentBlock];
  }
  covmat+=sig*sig.conjugate().transposed();
}

/*!
  Signals are corrected by factor 1/sqrt(distance to source)
*/
void SimulatedCrossSpectrum::addActiveVertical(const ActiveFKSteering& geom)
{
  int stationCount=_array.count();
  ComplexMatrix& covmat=*_matrix;
  ComplexMatrix sig(stationCount, 1);
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(is, 0);
    s=_stationSig[2][is][_currentBlock];
    s*=geom.amplitudeFactor(is);
  }
  covmat+=sig*sig.conjugateTransposedVector();
}

void SimulatedCrossSpectrum::addActiveRayleigh(const ActiveFKSteering& geom)
{
  int stationCount=_array.count();
  ComplexMatrix& covmat=*_matrix;
  ComplexMatrix sig(2*stationCount, 1);
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(is, 0);
    s=geom.radialRotation(is).sin()*_stationSig[1][is][_currentBlock];
    s+=geom.radialRotation(is).cos()*_stationSig[0][is][_currentBlock];
    s*=geom.amplitudeFactor(is);
  }
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(stationCount+is, 0);
    s=_stationSig[2][is][_currentBlock];
    s*=geom.amplitudeFactor(is);
  }
  covmat+=sig*sig.conjugateTransposedVector();
}

void SimulatedCrossSpectrum::addActiveTransverse(const ActiveFKSteering& geom)
{
  int stationCount=_array.count();
  ComplexMatrix& covmat=*_matrix;
  ComplexMatrix sig(stationCount, 1);
  for(int is=0; is<stationCount; is++) {
    Complex& s=sig.at(is, 0);
    s=geom.transverseRotation(is).sin()*_stationSig[1][is][_currentBlock];
    s+=geom.transverseRotation(is).cos()*_stationSig[0][is][_currentBlock];
    s*=geom.amplitudeFactor(is);
  }
  covmat+=sig*sig.conjugateTransposedVector();
}

void SimulatedCrossSpectrum::addSource()
{
  TRACE;
  int nSrc=_sourceSig.count();
  _sourceSig.resize(nSrc+1);
  _sourceSig[nSrc]=new SourceSignal;
}

void SimulatedCrossSpectrum::setSourceCount(int nSrc)
{
  TRACE;
  qDeleteAll(_sourceSig);
  _sourceSig.resize(nSrc);
  for(int i=0; i<nSrc; i++) {
    _sourceSig[i]=new SourceSignal;
  }
}

void SimulatedCrossSpectrum::removeSource(int index)
{
  TRACE;
  int nSrc=_sourceSig.count();
  if(index>=nSrc) return;
  delete _sourceSig[index];
  _sourceSig.remove(index);
}

void SimulatedCrossSpectrum::setSourceSignals(int iSrc, const SourceParameters& src,
                                              const MediumParameters& medium, int blockCount)
{
  TRACE;
  _sourceSig[iSrc]->setParameters(_waveModel, src, medium, blockCount, _array.relativePos());
}

void SimulatedCrossSpectrum::clearSignals()
{
  TRACE;
  int nStat=_array.count();
  if(_stationSig) {
    for(int iComp=0; iComp<3; iComp++) {
      for(int iStat=0; iStat<nStat; iStat++) {
        delete [] _stationSig[iComp][iStat];
      }
      delete [] _stationSig[iComp];
    }
    delete [] _stationSig;
  }
}

void SimulatedCrossSpectrum::setStationSignals(int blockCount,
                                               double verticalNoise,
                                               double horizontalNoise)
{
  TRACE;
  Q_UNUSED(verticalNoise);
  TODO_WARNING;
  clearSignals();
  int nStat=_array.count();
  int nSrc=_sourceSig.count();
  _stationSig=new Complex**[3];
  for(int iComp=0; iComp<3; iComp++) {
    _stationSig[iComp]=new Complex *[nStat];
    for(int iStat=0; iStat<nStat; iStat++) {
      _stationSig[iComp][iStat]=new Complex[blockCount];
    }
  }
  Angle sr;
  sr.setDegrees(_sensorRotation);
  Complex src, srcX, srcY;
  Complex noise;
  verticalNoise=0.0;   // Do not use this not purely decorrelated noise
  horizontalNoise=0.0;
  // The noise added with AbstractFKFunction::addIncoherentNoise(double horizontal, double vertical)

  for(int iSrc=0; iSrc<nSrc; iSrc++) {
    SourceSignal * sig=_sourceSig[iSrc];
    switch(sig->parameters().polarization()) {
    case Mode::Vertical:
      for(int iStat=0; iStat<nStat; iStat++) {
        for(int iBlock=0; iBlock<blockCount; iBlock++) {
          //noise.setExp(verticalNoise, _noisePhaseGenerator->uniform(0.0, 2.0*M_PI));
          _stationSig[2][iStat][iBlock]+=sig->stationSignal(iBlock, iStat)+noise;
        }
      }
      break;
    case Mode::Radial:
      for(int iStat=0; iStat<nStat; iStat++) {
        double a=sig->parameters().azimuth(_waveModel, _array.relativePos(iStat));
        double ca=cos(a);
        double sa=sin(a);
        for(int iBlock=0; iBlock<blockCount; iBlock++) {
          src=sig->stationSignal(iBlock, iStat);
          srcX=src*ca;
          srcY=src*sa;
          //noise.setExp(horizontalNoise, _noisePhaseGenerator->uniform(0.0, 2.0*M_PI));
          _stationSig[0][iStat][iBlock]+=srcX*sr.cos()+srcY*sr.sin()+noise;
          //noise.setExp(horizontalNoise, _noisePhaseGenerator->uniform(0.0, 2.0*M_PI));
          _stationSig[1][iStat][iBlock]+=srcY*sr.cos()-srcX*sr.sin()+noise;
        }
      }
      break;
    case Mode::Rayleigh: {
        Complex ellh(0.0, -sin(sig->parameters().ellipticity()));
        double ellv=cos(sig->parameters().ellipticity());
        for(int iStat=0; iStat<nStat; iStat++) {
          double a=sig->parameters().azimuth(_waveModel, _array.relativePos(iStat));
          double ca=cos(a);
          double sa=sin(a);
          for(int iBlock=0; iBlock<blockCount; iBlock++) {
            src=sig->stationSignal(iBlock, iStat);
            srcX=src*ca;
            srcY=src*sa;
            //noise.setExp(horizontalNoise, _noisePhaseGenerator->uniform(0.0, 2.0*M_PI));
            _stationSig[0][iStat][iBlock]+=(srcX*sr.cos()+srcY*sr.sin())*ellh+noise;
            //noise.setExp(horizontalNoise, _noisePhaseGenerator->uniform(0.0, 2.0*M_PI));
            _stationSig[1][iStat][iBlock]+=(srcY*sr.cos()-srcX*sr.sin())*ellh+noise;
            //noise.setExp(verticalNoise, _noisePhaseGenerator->uniform(0.0, 2.0*M_PI));
            _stationSig[2][iStat][iBlock]+=src*ellv+noise;
          }
        }
      }
      break;
    case Mode::Transverse:
    case Mode::Love:
      for(int iStat=0; iStat<nStat; iStat++) {
        double a=sig->parameters().azimuth(_waveModel, _array.relativePos(iStat));
        double ca=cos(a);
        double sa=-sin(a);
        for(int iBlock=0; iBlock<blockCount; iBlock++) {
          src=sig->stationSignal(iBlock, iStat);
          srcX=src*sa;
          srcY=src*ca;
          noise.setExp(horizontalNoise, _noisePhaseGenerator->uniform(0.0, 2.0*M_PI));
          _stationSig[0][iStat][iBlock]+=srcX*sr.cos()+srcY*sr.sin()+noise;
          //noise.setExp(horizontalNoise, _noisePhaseGenerator->uniform(0.0, 2.0*M_PI));
          _stationSig[1][iStat][iBlock]+=srcY*sr.cos()-srcX*sr.sin()+noise;
        }
      }
      break;
    }
  }
}
