/***************************************************************************
**
**  This file is part of figue.
**
**  figue is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  figue is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2005-07-27
**  Copyright: 2005-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <SciFigs.h>

#include "figueVersion.h"
#include "MainWindow.h"

MainWindow::MainWindow(QWidget *parent, Qt::WindowFlags f)
    : QMainWindow(parent, f)
{
  TRACE;
  resize(QSize(778, 533).expandedTo(minimumSizeHint()));
  setWindowTitle(tr("figue"));
  setWindowIcon(QIcon(":figue-48x48.png"));
  _sheet=new GraphicSheet(this);
  _sheet->setObjectName("sheet");
  setCentralWidget(_sheet);

  UpdateIcon * liveUpdate=new UpdateIcon(statusBar());
  liveUpdate->setName("figue");
  liveUpdate->setVersion(FIGUE_VERSION);
  liveUpdate->setVersionType(FIGUE_VERSION_TYPE);
  statusBar() ->addPermanentWidget(liveUpdate, 0);

  _sheet->setStatusBar(statusBar());

  statusBar()->setSizeGripEnabled(true);
  statusBar()->showMessage(tr("Ready"), 2000);
  connect(_sheet, SIGNAL(currentFileChanged(const QString&)),
          this, SLOT(currentFileChanged(const QString&)));
  addActions();

  liveUpdate->getLatestVersions();
}

void MainWindow::addActions()
{
  TRACE;
  QAction * a;
  QMenu * m;
  QToolBar * tb;

  // File menu and toolbar
  m=menuBar()->addMenu(tr("&File"));
  tb=addToolBar(tr("File"));
  _sheet->addMenu(m);
  _sheet->addFileActions(m, tb, true);

  m->addSeparator();

  a=new QAction(tr("&Quit"), this);
  a->setShortcut(tr("Ctrl+Q"));
  connect(a, SIGNAL(triggered()), this, SLOT(fileExit()));
  m->addAction(a);

  // Edit menu and toolbar
  m=menuBar()->addMenu(tr("&Edit"));
  tb=addToolBar(tr("Edit"));
  _sheet->addMenu(m);
  _sheet->addEditActions(m, tb, true);

  // Insert menu and toolbar
  m=menuBar()->addMenu(tr("&Insert"));
  tb=addToolBar(tr("Insert"));
  _sheet->addMenu(m);
  _sheet->addInsertActions(m, tb, true);

  // Format menu and toolbar
  m=menuBar()->addMenu(tr("&Format"));
  _sheet->addMenu(m);
  _sheet->addFormatActions(m, true);

  // Help menu
  m=menuBar()->addMenu(tr("&Help"));
  _sheet->addMenu(m);

  a=new QAction(tr("Online figue &Documentation"), this);
  a->setShortcut(tr("F1"));
  a->setStatusTip(tr("Access to online html documentation"));
  connect(a, SIGNAL(triggered()), this, SLOT(helpDocumentation()));
  m->addAction(a);

  a=QWhatsThis::createAction(this);
  m->addAction(a);

  m->addSeparator();

  a = new QAction(tr("&About"), this);
  a->setStatusTip(tr("Show figue's About box"));
  connect(a, SIGNAL(triggered()), this, SLOT(about()));
  m->addAction(a);

  a=new QAction(tr("About &Qt"), this);
  a->setStatusTip(tr("Show the Qt library's About box"));
  connect(a, SIGNAL(triggered()), qApp, SLOT(aboutQt()));
  m->addAction(a);
}

void MainWindow::closeEvent (QCloseEvent * e)
{
  TRACE;
  qApp->quit();
  e->ignore();
}

void MainWindow::fileExit()
{
  TRACE;
  qApp->quit();
}

void MainWindow::currentFileChanged(const QString& fileName)
{
  TRACE;
  setWindowTitle("figue - " + fileName);
}

void MainWindow::helpDocumentation()
{
  TRACE;
  QDesktopServices::openUrl(QUrl("http://www.geopsy.org/wiki/index.php/SciFigs:_Figue"));
}

void MainWindow::about()
{
  TRACE;
  QString version("Version: ");
  const PackageInfo * info=PackageInfo::package("figue");
  version+=info->version();
  if(!info->distribution().isEmpty()) {
    version+=" (";
    version+=info->distribution();
    version+=")";
  }
  Dialog * d=new Dialog(this);
  d->setWindowTitle(tr("About figue"));
  d->setMinimumSize(500,300);
  QLabel * text=new QLabel;
  text->setTextFormat(Qt::RichText);
  text->setWordWrap(true);
  text->setOpenExternalLinks(true);
  text->setText(tr("<p>%1</p>"
                   "<p>Figue is a graphical tool to modify, fix, customize, polish... all figures copied from "
                   "processing softwares such as Geopsy or Dinver. Additionally, Figue can help you create "
                   "plots from data.</p>"
                   "<p>Using, Copying and Modifying this program is granted to everyone under the terms "
                   "of the GNU Public License version 2. <b>However, we would appreciate that you properly "
                   "reference this work, released for free, in all your publications or reports achieved "
                   "with this software.</b></p>"
                   "<p>This program is provided AS IS with NO WARRANTY OF ANY KIND, "
                   "INCLUDING THE WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS "
                   "FOR A PARTICULAR PURPOSE.</p>").arg(version));
  d->setMainWidget(text, Dialog::Close);
  d->exec();
  delete d;
}
