/***************************************************************************
**
**  This file is part of dinver.
**
**  dinver is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  dinver is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2004-10-26
**  Copyright: 2004-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QTabWidget>
#include <QLabel>
#include <QLineEdit>
#include <QComboBox>

#include <DinverCore.h>
#include <QGpGuiTools.h>
#include <SciFigs.h>
#include "NAModelsPlot.h"
#include "PSViewer.h"
#include "NAModelsPlotProperties.h"

/*!
  \class NAModelsPlot NAModelsPlot.h
  \brief A NAModelsPlot is a layer to draw projection of a parameter space

  No data is directly stored in this object. It access directly to models generated by NA engine.

  The layer is exported either as a NAModelsPlot (for psViewer only) or as a XYValuePlot (available in Figue)

*/

const QString NAModelsPlot::xmlNAModelsPlotTag="NAModelsPlot";

REGISTER_GRAPHCONTENTLAYER(NAModelsPlot, "NAModelsPlot" );

NAModelsPlot::NAModelsPlot(AxisWindow * parent) :
    GraphContentsLayer(parent)
{
  TRACE;
  _psViewer=0;
  if(parent) setParent(graphContents());

  _size=0.1;
  _paramX=-1;
  _paramY=0;
}

NAModelsPlot::~NAModelsPlot()
{
  TRACE;
}

void NAModelsPlot::setParent(GraphContents * parent)
{
  TRACE;
  if(parent) {
    AxisWindow * w=parent->graph();
    if(w->sheet()) {
      _psViewer=qobject_cast<PSViewer *>(w->sheet()->parentWidget());
    }
  }
  GraphContentsLayer::setParent(parent);
}

void NAModelsPlot::paintData(const LayerPainterRequest& lp, QPainter& p, double dotpercm) const
{
  TRACE;
  _psViewer->lockDataModels();
  const GraphContentsOptions& gc=lp.options();
  ColorMap map=*_psViewer->colorMap();

  double pointSizeF=_size * dotpercm;
  int pointSize, halfSize;
  if(pointSizeF<2.0) {
    pointSize=2;
    halfSize=1;
  } else {
    pointSize=qRound(pointSizeF);
    halfSize=qRound(0.5*pointSizeF);
  }
  double x, y, misfit;
  ModelThreadList::const_iterator it;
  ModelThreadList::const_iterator itEnd=_psViewer->modelList().end();
  QColor col;
  for(it=_psViewer->modelList().begin();it!=itEnd;++it) {
    if(lp.terminated()) {
      _psViewer->unlockDataModels();
      return;
    }
    paramValue(x, y, misfit, it);
    if(x>gc.xVisMin() && x<gc.xVisMax()) {
      if(y>gc.yVisMin() && y<gc.yVisMax()) {
        guiColor(map.color(misfit), col);
        p.setPen(col);
        p.setBrush(col);
        int orx=gc.xr2s(x)-halfSize;
        int ory=gc.yr2s(y)-halfSize;
        p.drawEllipse(orx, ory, pointSize, pointSize);
      }
    }
  }
  p.setPen(Qt::black);
  p.setBrush(Qt::NoBrush);
  ViewerThreadMap::const_iterator itThread;
  ViewerThreadMap::const_iterator itThreadEnd=_psViewer->threads().end();
  for(itThread=_psViewer->threads().begin(); itThread!=itThreadEnd; ++itThread) {
    const InversionThread * t=itThread.key();
    BestIndex end;
    // Safe because all threads are already locked
    const BestModels * bestModels=t->bestModels();
    end.setValue(bestModels->count());
    for(BestIndex i; i<end; i++) {
      paramValue(x, y, misfit, t, bestModels->setIndex(i));
      if(x>gc.xVisMin() && x<gc.xVisMax()) {
        if(y>gc.yVisMin() && y<gc.yVisMax()) {
          int orx=gc.xr2s(x)-halfSize;
          int ory=gc.yr2s(y)-halfSize;
          p.drawEllipse(orx, ory, pointSize, pointSize);
        }
      }
    }
  }
  _psViewer->unlockDataModels();
}

inline void NAModelsPlot::paramValue(double& x, double& y, double& misfit,
                                     const InversionThread * t,
                                     const SetIndex& index) const
{
  TRACE;
  misfit=t->misfit(index);
  if(_paramX>=0) {
    x=t->variableParameterValue(index, _paramX);
  } else {
    x=misfit;
  }
  if(_paramY>=0) {
    y=t->variableParameterValue(index, _paramY);
  } else {
    y=misfit;
  }
}

inline void NAModelsPlot::paramValue(double& x, double& y, double& misfit,
                                     ModelThreadList::const_iterator it) const
{
  TRACE;
  const ModelThreadInfo & modelInfo=*it;
  paramValue(x, y, misfit, modelInfo.thread(), modelInfo.index());
}

Rect NAModelsPlot::boundingRect() const
{
  TRACE;
  if(_psViewer->modelList().size() > 0) {
    _psViewer->lockDataModels();
    ModelThreadList::const_iterator it=_psViewer->modelList().begin();
    ModelThreadList::const_iterator itEnd=_psViewer->modelList().end();
    Rect r;
    double x, y, misfit;
    for(; it!=itEnd; ++it) {
      paramValue(x, y, misfit, it);
      if(!isinf(misfit)) {
        r.add(x, y);
      }
    }
    _psViewer->unlockDataModels();
    return r;
  } else return Rect();
}

uint NAModelsPlot::_tab=PropertyTab::uniqueId();

/*!
  Setup property editor
*/
void NAModelsPlot::addProperties(PropertyProxy * pp)
{
  TRACE;
  if(pp->setCurrentTab(_tab)) {
    pp->addReference(this);
  } else {
    NAModelsPlotProperties * w=new NAModelsPlotProperties;
    w->setParameterList(_psViewer->parameterList());
    pp->addTab(_tab, tr("Parameters"), w, this);
  }
}

/*!
  Clean property editor
*/
void NAModelsPlot::removeProperties(PropertyProxy * pp)
{
  pp->removeTab(_tab, this);
}

void NAModelsPlot::properties(PropertyWidget * w) const
{
  TRACE;
  w->setValue(NAModelsPlotProperties::DotDiameter, dotDiameter()*10.0);
  w->setValue(NAModelsPlotProperties::ParamX, paramX()+1);
  w->setValue(NAModelsPlotProperties::ParamY, paramY()+1);
}

void NAModelsPlot::setProperty(uint, int pid, QVariant val)
{
  TRACE;
  bool axisUpdate=false;
  switch(pid) {
  case NAModelsPlotProperties::DotDiameter:
    setDotDiameter(val.toDouble()*0.1);
    graphContents()->deepUpdate();
    break;
  case NAModelsPlotProperties::ParamX:
    setParamX(val.toInt()-1);
    axisUpdate=true;
    break;
  case NAModelsPlotProperties::ParamY:
    setParamY(val.toInt()-1);
    axisUpdate=true;
    break;
  }
  if(axisUpdate) {
    // force update here because it affects the axis as well (and set limits again)
    Rect r=boundingRect();
    graph()->xAxis()->setRange(r.x1(), r.x2());
    graph()->yAxis()->setRange(r.y1(), r.y2());
    graph() ->deepUpdate();
  }
}

void NAModelsPlot::setAxisProperties(int paramIndex, Axis * axis)
{
  QString str;
  if(paramIndex>=0) {
    const Parameter & param=*_psViewer->parameterList()->variableParameter(paramIndex);
    if(param.unit().isEmpty()) {
      str=param.name();
    } else {
      str=QString("%1 (%2)").arg(param.name()).arg(param.unit());
    }
    axis->setRange(param.minimum(), param.maximum());
  } else {
    str="Misfit";
    axis->setScaleType(Scale::Log);
    axis->setAutoTicks(false);
    axis->setMajorTicks(10.0);
    axis->setMinorTicks(2.0);
  }
  axis->setTitle(str);
}

/*!
  px may range from -1 to the number of available parameters

  -1 means misfit
*/
void NAModelsPlot::setParamX(int px)
{
  TRACE;
  LayerLocker ll(this);
  _paramX=px;
  if(_paramX>=_psViewer->parameterList()->variableParameterCount()) {
    App::log(tr("Parameter index %1 out or range\n").arg(_paramX));
    _paramX=-1;
  }
  setAxisProperties(_paramX, graphContents()->graph()->xAxis());
}

/*!
  py may range from -1 to the number of available parameters

  -1 means misfit
*/
void NAModelsPlot::setParamY(int py)
{
  TRACE;
  LayerLocker ll(this);
  _paramY=py;
  if(_paramY>=_psViewer->parameterList()->variableParameterCount()) {
    App::log(tr("Parameter index %1 out or range\n").arg(_paramY));
    _paramY=-1;
  }
  setAxisProperties(_paramY, graphContents()->graph()->yAxis());
}

bool NAModelsPlot::saveParameters() const
{
  TRACE;
  return _psViewer->saveParameters();
}

const QString& NAModelsPlot::xml_tagName() const
{
  TRACE;
  if(saveParameters())
    return xmlNAModelsPlotTag;
  else
    return XYValuePlot::xmlXYValuePlotTag;
}

void NAModelsPlot::xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const
{
  TRACE;
  if(saveParameters()) {
    writeProperty(s, "paramX", paramX());
    writeProperty(s, "paramY", paramY());
  }
  GraphContentsLayer::xml_writeProperties(s, context);
  XMLSciFigs * scifigsContext=static_cast<XMLSciFigs *>(context);
  if(scifigsContext->data() && !saveParameters()) writeBinaryData(s, context);
}

void NAModelsPlot::xml_writeChildren(XML_WRITECHILDREN_ARGS) const
{
  TRACE;
  XMLSciFigs * scifigsContext=static_cast<XMLSciFigs *>(context);
  if(scifigsContext->data() && !saveParameters()) {
    _psViewer->colorMap()->xml_save(s, context);
  }
}

void NAModelsPlot::xml_writeBinaryData(XML_WRITEBINARYDATA_ARGS) const
{
  TRACE;
  Q_UNUSED(context)
  _psViewer->lockDataModels();
  ModelThreadList::const_iterator it;
  ModelThreadList::const_iterator itEnd=_psViewer->modelList().end();
  int n=_psViewer->modelList().count();
  s << n;

  if(_paramX>=0) {
    int p=_paramX;
    for(it=_psViewer->modelList().begin();it!=itEnd;++it) {
      const ModelThreadInfo& modelInfo=*it;
      s << modelInfo.thread()->variableParameterValue(modelInfo.index(),p);
    }
  } else {
    for(it=_psViewer->modelList().begin();it!=itEnd;++it) {
      const ModelThreadInfo& modelInfo=*it;
      s << modelInfo.thread()->misfit(modelInfo.index());
    }
  }

  if(_paramY>=0) {
    int p=_paramY;
    for(it=_psViewer->modelList().begin();it!=itEnd;++it) {
      const ModelThreadInfo& modelInfo=*it;
      s << modelInfo.thread()->variableParameterValue(modelInfo.index(),p);
    }
  } else {
    for(it=_psViewer->modelList().begin();it!=itEnd;++it) {
      const ModelThreadInfo& modelInfo=*it;
      s << (*it).thread()->misfit(modelInfo.index());
    }
  }

  for(it=_psViewer->modelList().begin();it!=itEnd;++it) {
    const ModelThreadInfo& modelInfo=*it;
    s << (*it).thread()->misfit(modelInfo.index());
  }

  _psViewer->unlockDataModels();
}

XMLMember NAModelsPlot::xml_member(XML_MEMBER_ARGS)
{
  TRACE;
  if(saveParameters()) {
    if(tag=="paramX") return XMLMember(0);
    else if(tag=="paramY") return XMLMember(1);
  }
  return GraphContentsLayer::xml_member(tag, attributes, context)+2;
}


bool NAModelsPlot::xml_setProperty(XML_SETPROPERTY_ARGS)
{
  TRACE;
  bool ok=true;
  switch(memberID) {
  case 0: setParamX(content.toInt(ok)); return ok;
  case 1: setParamY(content.toInt(ok)); return ok;
  }
  return GraphContentsLayer::xml_setProperty(memberID-2, tag, attributes, content, context);
}
