/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-05-07
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef XYPLOT_H
#define XYPLOT_H

#include "GraphicObject.h"
#include "GraphContentsLayer.h"
#include "Symbol.h"

namespace SciFigs {

class XYPlotProperties;

class SCIFIGS_EXPORT XYPlot : public GraphContentsLayer
{
  Q_OBJECT
  Q_PROPERTY(double dotDiameter READ dotDiameter WRITE setDotDiameter SCRIPTABLE true)
  Q_PROPERTY(int count READ count STORED false SCRIPTABLE true)
  Q_PROPERTY(QString symbol READ symbolString WRITE setSymbol)
public:
  XYPlot(AxisWindow * parent=nullptr);
  ~XYPlot();

  virtual const QString& xml_tagName() const {return xmlXYPlotTag;}
  static const QString xmlXYPlotTag;

  int count() const {if(_xData) return _xData->count(); else return 0;}

  void setXData(VectorList<double> * x);
  void setYData(VectorList<double> * y);

  double dotDiameter() const {return _size;}
  void setDotDiameter (double d) {_size=d;}

  Symbol::Type symbol() const {return _symbol;}
  QString symbolString() const {return Symbol::convertType(_symbol);}
  void setSymbol(Symbol::Type s) {_symbol=s;}
  void setSymbol(const QString& s) {bool ok=true; _symbol=Symbol::convertType(s, ok);}

  virtual bool trackRectangle(int, double rx1, double ry1, double rx2, double ry2,
                               Qt::KeyboardModifiers eventState);
  virtual void select(VectorList<int> * indexList);
  // Calculate the rectangle that includes all the point
  virtual Rect boundingRect() const;

  virtual bool hasProperties() {return true;}
  virtual void addProperties(PropertyProxy * pp);
  virtual void removeProperties(PropertyProxy * pp);
  virtual void properties(PropertyWidget * w) const;
  virtual void setProperty(uint wid, int pid, QVariant val);
public slots:
  void swapXY();
protected:
  virtual void paintData(const LayerPainterRequest& lp, QPainter& p, double dotpercm) const;
  inline void getPaintParam(int& count, double& pointSize,
                            double& halfPointSize, double dotpercm) const;

  VectorList<double> * _xData;
  VectorList<double> * _yData;

  double _size;
  Symbol::Type _symbol;

  static uint _tab;

  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual void xml_writeBinaryData(XML_WRITEBINARYDATA_ARGS) const;
  virtual bool xml_setBinaryData(XML_SETBINARYDATA_ARGS);
};

inline void XYPlot::getPaintParam(int& count, double& pointSize,
                                  double& halfPointSize, double dotpercm) const
{
  TRACE;
  count=_xData->size();
  if(count>_yData->size()) {
    count=_yData->size();
  }
  pointSize=_size*dotpercm;
  halfPointSize=0.5*pointSize;
}

} // namespace SciFigs

#endif // XYPLOT_H
