/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-08-23
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef XUNIQUEYCOLORLINES_H
#define XUNIQUEYCOLORLINES_H

#include <QGpCoreMath.h>

#include "GraphContentsLayer.h"
#include "SciFigsDLLExport.h"
#include "Symbol.h"

namespace SciFigs {

class XYColorLinesProperties;

class SCIFIGS_EXPORT XUniqueYColorLines : public GraphContentsLayer
{
  Q_OBJECT
  Q_PROPERTY(double lineWeight READ lineWeight WRITE setLineWeight SCRIPTABLE true)
public:
  XUniqueYColorLines(AxisWindow * parent=nullptr);
  ~XUniqueYColorLines();

  virtual const QString& xml_tagName() const {return xmlXUniqueYColorLinesTag;}
  static const QString xmlXUniqueYColorLinesTag;

  void clear();
  void setPointCount(int nCurve, int nX);

  XUniqueYColorData * data() {return _data;}
  const XUniqueYColorData * data() const {return _data;}
  void setData(XUniqueYColorData * d);

  const double * x() const {return _data->x();}
  const double * y() const {return _data->y();}
  double * x() {return _data->x();}
  double * y() {return _data->y();}

  const Color * colors() const {return _data->colors();}
  void setColors(Color * col);
  void setColor(int index, const Color& col);
  void setSymbol(Symbol::Type s);

  // Calculate the rectangle that includes all the curves
  Rect boundingRect() const {return _data->boundingRect();}

  virtual bool trackRectangle(int, double rx1, double ry1, double rx2, double ry2, Qt::KeyboardModifiers);
  int curveCount() {return _data->curveCount();}
  int xCount() {return _data->xCount();}
  inline void setLineWeight(double lw) {_lineWeight=lw;}
  double lineWeight() const {return _lineWeight;}
  double lineWeightMM() const {return _lineWeight*10.0;}

  virtual bool hasProperties() {return true;}
  virtual void addProperties(PropertyProxy * pp);
  virtual void removeProperties(PropertyProxy * pp);
  virtual void properties(PropertyWidget * w) const;
  virtual void setProperty(uint wid, int pid, QVariant val);
signals:
#if(QT_VERSION>=QT_VERSION_CHECK(6, 0, 0))
  void dataSelected(int id, const QList<int> * indexList);
#else
  void dataSelected(int id, const QVector<int> * indexList);
#endif
protected:
  virtual void paintData(const LayerPainterRequest& lp, QPainter& p, double dotpercm) const;

  bool _dataOwner;
  XUniqueYColorData * _data;
  Symbol::Type _symbol;
  double _lineWeight;
  // Storage for pointer to tab widget
  XYColorLinesProperties * _propertiesTab;
protected:
  virtual void xml_writeProperties(XML_WRITEPROPERTIES_ARGS) const;
  virtual void xml_writeBinaryData(XML_WRITEBINARYDATA_ARGS) const;
  virtual bool xml_setBinaryData(XML_SETBINARYDATA_ARGS);
};

} // namespace SciFigs

#endif // XUNIQUEYCOLORLINES_H
