/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-04-02
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "GraphContentsLayerProperties.h"
#include "GraphContents.h"
#include "AxisWindow.h"
#include "XMLSciFigs.h"

namespace SciFigs {

/*!
  \class GraphContentsLayerProperties GraphContentsLayerProperties.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
GraphContentsLayerProperties::GraphContentsLayerProperties(QWidget * parent)
    : PropertyWidget(parent)
{
  TRACE;
  setupUi(this);

  layers->setModel(new LayerPropertiesItem(this));
  layers->setItemDelegate(new LayerPropertiesDelegate(this));
  _uniqueLayerStack=true;
}

GraphContentsLayerProperties::~GraphContentsLayerProperties()
{
}

LayerPropertiesItem * GraphContentsLayerProperties::layerModel() const
{
  return static_cast<LayerPropertiesItem *>(layers->model());
}

void GraphContentsLayerProperties::addGraph(GraphContents * g)
{
  TRACE;
  int i=_graphList.indexOf(g);
  if(i>=0) {
#if(QT_VERSION >= QT_VERSION_CHECK(5, 13, 0))
    _graphList.swapItemsAt(0, i);
#else
    qSwap(_graphList[0], _graphList[i]);
#endif
  } else {
    _graphList.prepend(g);
  }
  setGraphList();
  if(_uniqueLayerStack) {
    layerModel()->setGraphs(_graphList);
  } else {
    on_currentGraphCombo_currentIndexChanged(0);
  }
}

void GraphContentsLayerProperties::removeGraph(GraphContents * g)
{
  TRACE;
  int i=_graphList.indexOf(g);
  if(i<0) {
    return;
  }
  _graphList.removeAt(i);
  setGraphList();
  if(_uniqueLayerStack) {
    layerModel()->setGraphs(_graphList);
  } else {
    on_currentGraphCombo_currentIndexChanged(0);
  }
}

bool GraphContentsLayerProperties::graphLessThan(const GraphContents * g1, const GraphContents * g2)
{
  QString g1Name, g2Name;
  if(g1->graph()->objectName().isEmpty())
    g1Name="unamed_graph";
  else
    g1Name=g1->graph()->objectName();
  if(g2->graph()->objectName().isEmpty())
    g2Name="unamed_graph";
  else
    g2Name=g2->graph()->objectName();
  return g1Name<g2Name;
}

void GraphContentsLayerProperties::setGraphList()
{
  TRACE;
  currentGraphCombo->clear();
  if(!_graphList.isEmpty()) {
    GraphContents * first=_graphList.first();
    int nGraphs=_graphList.count();
    _uniqueLayerStack=true;
    for(int i=1; i<nGraphs; i++) {
      if(!first->hasSameLayers(*_graphList.at(i))) {
        _uniqueLayerStack=false;
        break;
      }
    }
    if(_uniqueLayerStack) {
      if(nGraphs==1) {
        if(_graphList.first()->graph()->objectName().isEmpty()) {
          currentGraphCombo->addItem(tr("unamed_graph"));
        } else {
          currentGraphCombo->addItem(_graphList.first()->graph()->objectName());
        }
      } else {
        currentGraphCombo->addItem(tr("%1 graphs with similar layers").arg(_graphList.count()));
      }
      currentGraphCombo->setEnabled(false);
    } else {
      std::sort(_graphList.begin(), _graphList.end(), graphLessThan);
      for(int i=0; i<nGraphs; i++) {
        GraphContents * g=_graphList.at(i);
        if(g->graph()->objectName().isEmpty())
          currentGraphCombo->addItem(tr("unamed_graph"));
        else
          currentGraphCombo->addItem(g->graph()->objectName());
      }
      currentGraphCombo->setEnabled(true);
      currentGraphCombo->setCurrentIndex(_graphList.indexOf(first));
    }
  }
}

GraphContents * GraphContentsLayerProperties::currentGraph() const
{
  TRACE;
  int i=currentGraphCombo->currentIndex();
  if(i<_graphList.count() && i>=0) {
    return _graphList[i];
  } else {
    return nullptr;
  }
}

void GraphContentsLayerProperties::reset()
{
  TRACE;
  if(_uniqueLayerStack) {
    layerModel()->setGraphs(_graphList);
  } else {
    layerModel()->setGraph(currentGraph());
  }
  layers->resizeColumnsToContents();
}

void GraphContentsLayerProperties::on_currentGraphCombo_currentIndexChanged(int)
{
  TRACE;
  if(!_uniqueLayerStack) {
    layerModel()->setGraph(currentGraph());
  }
}

void GraphContentsLayerProperties::on_saveLayers_clicked()
{
  TRACE;
  GraphContents * selectedGraph=currentGraph();
  QModelIndexList l=layers->selectionModel()->selectedIndexes();
  if(l.isEmpty()) {
    selectedGraph->saveLayers();
  } else {
    LayerPropertiesItem * model=layerModel();
    model->selectAllLayers(false);
    model->selectLayers(l);
    selectedGraph->saveLayers();
    model->selectAllLayers(true);
  }
}

void GraphContentsLayerProperties::on_copyLayers_clicked()
{
  TRACE;
  GraphContents * selectedGraph=currentGraph();
  QModelIndexList l=layers->selectionModel()->selectedIndexes();
  if(l.isEmpty()) {
    selectedGraph->copyLayers();
  } else {
    LayerPropertiesItem * model=layerModel();
    model->selectAllLayers(false);
    model->selectLayers(l);
    selectedGraph->copyLayers();
    model->selectAllLayers(true);
  }
}

void GraphContentsLayerProperties::on_appendLayers_clicked()
{
  TRACE;
  if(_uniqueLayerStack) {
    QString fileName=Message::getOpenFileName(tr("Append layers"), tr("Graph layers (*.layer)"));
    if(!fileName.isEmpty()) {
      for(int i=_graphList.count()-1; i>=0; i--) {
        _graphList.at(i)->appendLayers(fileName);
      }
    }
  } else {
    currentGraph()->appendLayers();
  }
  reset();
}

void GraphContentsLayerProperties::on_prependLayers_clicked()
{
  TRACE;
  if(_uniqueLayerStack) {
    QString fileName=Message::getOpenFileName(tr("Prepend layers"), tr("Graph layers (*.layer)"));
    if(!fileName.isEmpty()) {
      for(int i=_graphList.count()-1; i>=0; i--) {
        _graphList.at(i)->prependLayers(fileName);
      }
    }
  } else {
    currentGraph()->prependLayers();
  }
  reset();
}

void GraphContentsLayerProperties::on_removeLayers_clicked()
{
  TRACE;
  QModelIndexList l=layers->selectionModel()->selectedRows();
  if(l.count()>0) {
    if(Message::question(MSG_ID, tr("Remove layers"),
                                 tr("Do you want to remove all selected layers(%1)?").arg(l.count()),
                                 Message::yes(), Message::no())==Message::Answer1) {
      return;
    }
    int nRemoved=layerModel()->removeLayers(l);
    reset();
    if(nRemoved<l.count()) {
      Message::information(MSG_ID, tr("Remove layers"),
                                   tr("%1 layers are not removable.").arg(l.count()-nRemoved));
    }
  }
}

void GraphContentsLayerProperties::on_mergeLayers_clicked()
{
  TRACE;
  LayerPropertiesItem * model=layerModel();
  model->selectAllLayers(false);
  QModelIndexList l=layers->selectionModel()->selectedRows();
  model->selectLayers(l);
  if(_uniqueLayerStack) {
    for(int i=_graphList.count()-1; i>=0; i--) {
      _graphList.at(i)->mergeLayers();
    }
    reset();
  } else {
    GraphContents * selectedGraph=currentGraph();
    if(selectedGraph&& selectedGraph->mergeLayers()) {
      reset();
    }
  }
  model->selectAllLayers(true);
}

void GraphContentsLayerProperties::on_moveLayerUp_clicked()
{
  TRACE;
  QModelIndexList l=layers->selectionModel()->selectedRows();
  if(l.count()>0) {
    QList<GraphContentsLayer *> list;
    LayerPropertiesItem * model=layerModel();
    list=model->moveLayersUp(l);
    select(model, list);
  }
}

void GraphContentsLayerProperties::on_moveLayerDown_clicked()
{
  TRACE;
  QModelIndexList l=layers->selectionModel()->selectedRows();
  if(l.count()>0) {
    QList<GraphContentsLayer *> list;
    LayerPropertiesItem * model=layerModel();
    list=model->moveLayersDown(l);
    select(model, list);
  }
}

void GraphContentsLayerProperties::select(LayerPropertiesItem * model, const QList<GraphContentsLayer *>& list)
{
  TRACE;
  if(!list.isEmpty()) {
    int n=list.count();
    for(int i=0; i<n; i++) {
      QModelIndex index=model->indexOf(list.at(i));
      if(i==0) {
        layers->selectionModel()->setCurrentIndex(index, QItemSelectionModel::Rows);
      }
      layers->selectionModel()->select(index,
                                       QItemSelectionModel::Select |
                                       QItemSelectionModel::Rows);
    }
  }
}

void LayerPropertiesItem::setGraph(GraphContents * g)
{
  TRACE;
  beginResetModel();
  _graphs.clear();
  if(g) {
    _graphs.append(g);
  }
  endResetModel();
}

void LayerPropertiesItem::setGraphs(const QList<GraphContents *>& l)
{
  TRACE;
  beginResetModel();
  _graphs=l;
  endResetModel();
}

int LayerPropertiesItem::rowCount(const QModelIndex &) const
{
  TRACE;
  if(_graphs.isEmpty()) {
    return 0;
  } else {
    return _graphs.first()->layerCount();
  }
}

int LayerPropertiesItem::columnCount(const QModelIndex & ) const
{
  TRACE;
  return 4;
}

/*!
  Only the first GraphContents is considered.
*/
QModelIndex LayerPropertiesItem::indexOf(GraphContentsLayer * l) const
{
  TRACE;
  return index(_graphs.first()->layerCount()-_graphs.first()->indexOf(l)-1, 0);
}

int LayerPropertiesItem::layerIndex(const GraphContents * graph, const QModelIndex &index)
{
  return graph->layerCount()-index.row()-1;
}

const GraphContentsLayer * LayerPropertiesItem::layer(const GraphContents * graph, const QModelIndex &index) const
{
  return graph->layer(layerIndex(graph, index));
}

GraphContentsLayer * LayerPropertiesItem::layer(GraphContents * graph, const QModelIndex &index)
{
  return graph->layer(layerIndex(graph, index));
}

QVariant LayerPropertiesItem::data(const QModelIndex &index, int role) const
{
  TRACE;
  const GraphContentsLayer * l=layer(_graphs.first(), index);
  switch (role) {
  case Qt::DisplayRole:
    switch(index.column()) {
    case 0: return l->opacity();
    case 1: return l->objectName();
    case 2: return l->xml_tagName();
    default: return l->comments();
    }
  case Qt::ToolTipRole:
    return l->comments();
  case Qt::ForegroundRole:
    switch(index.column()) {
    case 0: return isOpacityConstant(index) ? QVariant() : QColor(Qt::red);
    case 1: return isNameConstant(index) ? QVariant() : QColor(Qt::red);
    case 3: return isCommentConstant(index) ? QVariant() : QColor(Qt::red);
    default: return QVariant();
    }
  default:
    return QVariant();
  }
}

bool LayerPropertiesItem::setData (const QModelIndex & index, const QVariant & value, int role)
{
  TRACE;
  for(int i=_graphs.count()-1; i>=0; i--) {
    GraphContentsLayer * l=layer(_graphs.at(i), index);
    switch (role) {
    case Qt::EditRole:
      switch(index.column()) {
      case 0:
        l->setOpacity(value.toDouble());
        l->deepUpdate();
        break;
      case 1:
        l->setObjectName(value.toString());
        break;
      case 3:
        l->setComments(value.toString());
        break;
      default:
        return false;
      }
      break;
    default:
      return false;
    }
  }
  return true;
}

QVariant LayerPropertiesItem::headerData(int section, Qt::Orientation orientation, int role) const
{
  TRACE;
  if(role!=Qt::DisplayRole) return QVariant();
  if(orientation==Qt::Horizontal) {
    switch(section) {
    case 0: return tr("Opacity");
    case 1: return tr("Name");
    case 2: return tr("Type");
    default: return tr("Comments");
    }
  } else {
    return _graphs.first()->layerCount()-section;
  }
}

Qt::ItemFlags LayerPropertiesItem::flags (const QModelIndex & index) const
{
  TRACE;
  if(index.column()==2) {
    return Qt::ItemIsSelectable | Qt::ItemIsEnabled;
  } else {
    return Qt::ItemIsSelectable | Qt::ItemIsEnabled | Qt::ItemIsEditable;
  }
}

int LayerPropertiesItem::removeLayers(QModelIndexList l)
{
  TRACE;
  // First convert index list to layer pointers
  QModelIndex index;
  int nLayers=0;
  beginResetModel();
  for(int i=_graphs.count()-1; i>=0; i--) {
    GraphContents * graph=_graphs.at(i);
    QList<GraphContentsLayer *> layers;
    foreach(index, l) {
      GraphContentsLayer * l=layer(graph, index);
      if(l->isRemovable()) {
        layers.append(l);
      }
    }
    nLayers=layers.count();
    for(int i=0; i<nLayers; i++) {
      graph->removeLayer(layers.at(i));
    }
  }
  endResetModel();
  return nLayers;
}

QList<GraphContentsLayer *> LayerPropertiesItem::moveLayersUp(QModelIndexList l)
{
  TRACE;
  QList<int> indexList;
  QModelIndex index;
  QList<GraphContentsLayer *> layers;
  foreach(index, l) {
    indexList << layerIndex(_graphs.first(), index);
  }
  std::sort(indexList.begin(), indexList.end());
  if(indexList.last()==_graphs.first()->layerCount()-1) {
    return layers; // Selected layers are already on top
  }
  // Transform into a ptr list, sorted from top to bottom
  beginResetModel();
  for(int i=_graphs.count()-1; i>=0; i--) {
    GraphContents * graph=_graphs.at(i);
    int n=indexList.count();
    layers.clear();
    for(int i=n-1; i>=0; i--) {
      layers << graph->layer(indexList.at(i));
    }
    for(int i=0; i<n; i++) {
      graph->moveLayerUp(layers.at(i));
    }
  }
  endResetModel();
  return layers;
}

QList<GraphContentsLayer *> LayerPropertiesItem::moveLayersDown(QModelIndexList l)
{
  TRACE;
  QList<int> indexList;
  QModelIndex index;
  QList<GraphContentsLayer *> layers;
  foreach(index, l) {
    indexList << layerIndex(_graphs.first(), index);
  }
  std::sort(indexList.begin(), indexList.end());
  if(indexList.first()==0) {
    return layers; // Selected layers are already on bottom
  }
  // Transform into a ptr list, sorted from bottom to top
  beginResetModel();
  for(int i=_graphs.count()-1; i>=0; i--) {
    GraphContents * graph=_graphs.at(i);
    int n=indexList.count();
    layers.clear();
    for(int i=0; i<n; i++) {
      layers << graph->layer(indexList.at(i));
    }
    for(int i=0; i<n; i++) {
      graph->moveLayerDown(layers.at(i));
    }
  }
  endResetModel();
  return layers;
}

void LayerPropertiesItem::selectLayers(QModelIndexList l)
{
  TRACE;
  QModelIndex index;
  for(int i=_graphs.count()-1; i>=0; i--) {
    GraphContents * graph=_graphs.at(i);
    foreach(index, l) {
      graph->layer(layerIndex(graph, index))->setSelected(true);
    }
  }
}

void LayerPropertiesItem::selectAllLayers(bool s)
{
  for(int i=_graphs.count()-1; i>=0; i--) {
    _graphs.at(i)->selectAllLayers(s);
  }
}

bool LayerPropertiesItem::isNameConstant(const QModelIndex & index) const
{
  int lIndex=layerIndex(_graphs.first(), index);
  QString first=_graphs.first()->layer(lIndex)->objectName();
  for(int i=_graphs.count()-1; i>=0; i--) {
    if(_graphs.at(i)->layer(lIndex)->objectName()!=first) {
      return false;
    }
  }
  return true;
}

bool LayerPropertiesItem::isOpacityConstant(const QModelIndex & index) const
{
  int lIndex=layerIndex(_graphs.first(), index);
  double first=_graphs.first()->layer(lIndex)->opacity();
  for(int i=_graphs.count()-1; i>=0; i--) {
    if(_graphs.at(i)->layer(lIndex)->opacity()!=first) {
      return false;
    }
  }
  return true;
}

bool LayerPropertiesItem::isCommentConstant(const QModelIndex & index) const
{
  int lIndex=layerIndex(_graphs.first(), index);
  QString first=_graphs.first()->layer(lIndex)->comments();
  for(int i=_graphs.count()-1; i>=0; i--) {
    if(_graphs.at(i)->layer(lIndex)->comments()!=first) {
      return false;
    }
  }
  return true;
}

QWidget * LayerPropertiesDelegate::createEditor(QWidget * parent, const QStyleOptionViewItem &,
    const QModelIndex &index) const
{
  TRACE;
  ASSERT(qobject_cast<const LayerPropertiesItem *>(index.model()));

  switch (index.column()) {
  case 0: {
      DoubleSpinBox * w=createDoubleSpinBox(parent);
      w->setMinimum(0.0);
      w->setMaximum(1.0);
      w->setSingleStep(0.05);
      connect(w,SIGNAL(valueChanged(double)), this, SLOT(commitWidget()));
      return w;
    }
  default:
    return createLineEdit(parent);
  }
}

} // namespace SciFigs
