/***************************************************************************
**
**  This file is part of SciFigs.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2002-05-27
**  Copyright: 2002-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "ColorMapTable.h"

namespace SciFigs {

ColorMapTable::ColorMapTable(QWidget *parent) :
    QScrollArea(parent)
{
  TRACE;
  setWidgetColor(viewport(), Qt::gray);
  setWidget(new QWidget(this) );
}

/*!
  Linked to properties: if \a c is true, the palette is the same for all
  palette widgets controled by this table. This is just a visual effect.
*/
void ColorMapTable::setConstant(bool c)
{
  QPalette palette;
  palette.setColor(QPalette::Window, c ? Qt::gray : QColor(164,112,128));
  viewport()->setPalette(palette);
}

void ColorMapTable::resizeEvent(QResizeEvent * e)
{
  TRACE;
  QScrollArea::resizeEvent(e);
  int nCol=_colors.count();
  int w=viewport() ->width();
  if(w<160) w=160;
  widget()->resize(w, nCol*29+10);
  nCol--;
  for(int iCol=0; iCol<nCol; iCol++) {
    _values.at(iCol) ->resize(w-106, 23);
  }
}

void ColorMapTable::setColorCount(int newNumber)
{
  TRACE;
  if(newNumber < 2) newNumber=2;
  int w=viewport() ->width();
  if(w < 160) w=160;
  widget() ->resize(w, 10 + newNumber * 29);

  int colorNumber=_colors.count();
  QLineEdit * lastEdit=0;
  ColorButton * lastButton=0;
  ColorButton * pb;
  QLineEdit * le;
  QString lastValue="0.0";
  QColor lastColor=Qt::red;
  if(colorNumber > 1) {
    lastButton=_colors.last();
    lastEdit=_values.last();
    lastColor=lastButton->color();
    lastValue=lastEdit->text();
  } else {
    //add first color to initialize widget list
    pb=new ColorButton(widget());
    pb->setGeometry(10, 10, 80, 23);
    pb->setColor(Qt::red);
    pb->setText( "1" );
    connect(pb, SIGNAL(colorChanged(QColor)), this, SIGNAL(changed()));
    colorNumber=1;
    lastButton=pb;
    pb->show();
    _colors.append(pb);
  }
  if(newNumber > colorNumber) {
    // need to add (newNumber-colorNumber) colors (lineedit & pushbutton) to the viewport
    for(int i=colorNumber;i < newNumber;i++ ) {
      le=new QLineEdit(widget());
      le->setAlignment(Qt::AlignRight);
      le->setGeometry(96, i * 29 - 5, w - 106, 23);
      le->setText(lastValue);
      if(lastEdit) setTabOrder(lastEdit, le);
      lastEdit=le;
      connect(le, SIGNAL(textChanged( const QString& )), this, SIGNAL(changed()));
      le->show();
      _values.append(le);

      pb=new ColorButton(widget());
      pb->setGeometry(10, 10 + i * 29, 80, 23);
      pb->setColor(lastColor);
      pb->setText(QString::number(i+1));
      connect(pb, SIGNAL(colorChanged(QColor)), this, SIGNAL(changed()) );
      setTabOrder(lastButton, pb);
      lastButton=pb;
      pb->show();
      _colors.append(pb);
    }
    emit changed();
  } else if(newNumber < colorNumber) {
    // need to remove (colorNumber-newNumber) colors (lineedit & pushbutton) from the viewport
    for(int i=colorNumber - 1;i >= newNumber;i-- ) {
      delete _colors.at(i);
      _colors.removeAt(i);
      delete _values.at(i - 1);
      _values.removeAt(i - 1);
    }
    _values.last()->setText(lastValue);
    _colors.last()->setColor(lastColor);
    emit changed();
  }
}

void ColorMapTable::setColors(const ColorMap& pal)
{
  TRACE;
  int ncolors=pal.count();
  setColorCount(ncolors);
  QColor col;
  for(int i=0; i<ncolors; i++) {
    guiColor(pal.color(i), col);
    _colors.at(i)->setColor(col);
  }
  emit changed();
}

void ColorMapTable::setValues(const ColorMap& pal)
{
  TRACE;
  int ncolors=pal.count();
  setColorCount(ncolors);
  for(int i=ncolors-2; i>=0; i--) {
    _values.at(i)->setText(QString::number(pal.upperValue(i)));
  }
  emit changed();
}

double ColorMapTable::value(int i) const
{
  TRACE;
  bool ok;
  if(i<0) return i=0;
  else if(i>=_values.count()) {
    i=_values.count()-1;
    if(i<0) return 0.0;
  }
  double val=_values.at(i)->text().toDouble(&ok);
  if( !ok) val=0;
  return val;
}

ColorMap ColorMapTable::colorMap() const
{
  TRACE;
  ColorMap map;
  int n=_colors.count();
  map.resize(n);

  double val, lastVal=0;
  bool ok;
  coreColor(_colors.first()->color(), map.color(0));
  for(int i=1; i<n; i++) {
    val=_values.at(i-1)->text().toDouble(&ok);
    if(!ok) {
      val=lastVal+1;
    }
    map.setUpperValue(i-1, val);
    coreColor(_colors.at(i)->color(), map.color(i));
    lastVal=val;
  }
  return map;
}

} // namespace SciFigs
