/***************************************************************************
**
**  This file is part of QGpGuiTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-03-31
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "PropertyProxy.h"
#include "PropertyCategory.h"
#include "PropertyTab.h"
#include "PropertyWidget.h"
#include "PropertyEditor.h"

namespace QGpGuiTools {

  /*!
    \class PropertyProxy PropertyProxy.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  PropertyProxy::PropertyProxy(QObject * parent)
      : QObject(parent)
  {
    _currentCategory=nullptr;
    _currentTab=nullptr;
    _blockUpdates=false;
    _categories=new PropertyCategory *[PropertyCategory::idCount()];
    for(uint i=0; i<PropertyCategory::idCount(); i++) {
      _categories[i]=nullptr;
    }
  }

  /*!
    Description of destructor still missing
  */
  PropertyProxy::~PropertyProxy()
  {
    if(isActive()) { // Remove all remaining docked widgets
      PropertyEditor::instance()->saveStates();
      PropertyEditor::instance()->takeWidgets();
      PropertyEditor::instance()->hide();
    }
    for(uint i=0; i<PropertyCategory::idCount(); i++) {
      delete _categories[i];
    }
    delete [] _categories;
  }

  void PropertyProxy::blockUpdates(bool b)
  {
    if(_blockUpdates) {
      if(!b) {
        _blockUpdates=b;
        setValues();
        restoreStates();
      }
    } else {
      _blockUpdates=b;
    }
  }

  void PropertyProxy::saveStates()
  {
    if(_blockUpdates) return;
    if(isActive()) {
      PropertyEditor::instance()->saveStates();
    }
  }

  void PropertyProxy::restoreStates()
  {
    if(_blockUpdates) return;
    if(isActive()) {
      PropertyEditor::instance()->restoreStates();
    }
  }

  /*!
    Set widget controls from object current values.
  */
  void PropertyProxy::setValues()
  {
    if(_blockUpdates) return;
    for(uint i=0; i<PropertyCategory::idCount(); i++) {
      PropertyCategory * pc=_categories[i];
      if(pc) {
        pc->setValues();
      }
    }
  }

  void PropertyProxy::setCurrentTabValues()
  {
    if(_blockUpdates) return;
    _currentCategory->setValues();
  }

  inline bool PropertyProxy::isActive() const
  {
    return PropertyEditor::instance() && PropertyEditor::instance()->proxy()==this;
  }

  void PropertyProxy::raiseEditor(uint category)
  {
    PropertyEditor * editor=PropertyEditor::instance();
    if(!editor) {
      editor=new PropertyEditor;
      editor->setObjectName("propertyEditor");
    }
    editor->setUpdatesEnabled(false);
    editor->setProxy(this);
    editor->show();
    editor->raise();
    if(_categories[category]) {
      editor->setCurrentTab(_categories[category]->caption());
    }
    editor->setUpdatesEnabled(true);
  }

  void PropertyProxy::setEditor()
  {
    for(uint i=0; i<PropertyCategory::idCount(); i++) {
      PropertyCategory * pc=_categories[i];
      if(pc) {
        ExpandTabWidget * w=new ExpandTabWidget;
        pc->setEditor(w);
        w->setTitlePrefix(pc->caption() + "::" );
        PropertyEditor::instance()->addCategory(pc->icon(), pc->caption(), w);
      }
    }
  }

  bool PropertyProxy::setCurrentCategory(uint category)
  {
    if(_categories[category]) {
      _currentCategory=_categories[category];
      return true;
    } else {
      _currentCategory=nullptr;
      return false;
    }
  }

  bool PropertyProxy::setCurrentTab(uint tab)
  {
    if(_currentCategory) {
      _currentTab=_currentCategory->find(tab);
      return _currentTab;
    } else return false;
  }

  PropertyWidget * PropertyProxy::currentTabWidget() const
  {
    return _currentTab ? _currentTab->widget() : 0;
  }

  void PropertyProxy::addCategory(uint category, QString caption, QIcon icon)
  {
    _currentCategory=new PropertyCategory;
    _currentCategory->setId(category);
    _currentCategory->setCaption(caption);
    _currentCategory->setIcon(icon);
    ASSERT(!_categories[category]);
    _categories[category]=_currentCategory;
    if(isActive()) {
      ExpandTabWidget * w=new ExpandTabWidget;
      w->setTitlePrefix(caption + "::" );
      PropertyEditor::instance()->addCategory(icon, caption, w);
    }
  }

  void PropertyProxy::addTab(uint tab, QString title, PropertyWidget * w, PropertyItem * item)
  {
    if(_currentCategory) {
      w->setId(tab);
      _currentTab=_currentCategory->addTab(tab, title, w, item);
      if(isActive()) {
        PropertyEditor::instance()->addTab(_currentCategory->caption(), title, w);
      }
    }
  }

  void PropertyProxy::addReference(PropertyItem * item)
  {
    if(_currentTab) _currentTab->addReference(item);
  }

  /*!
    Returns true if current tab has references.
  */
  bool PropertyProxy::hasReference() const
  {
    if(_currentTab) {
      return _currentTab->hasReference();
    } else {
      return false;
    }
  }

  /*!
    Returns the number of references to the current tab
  */
  int PropertyProxy::referenceCount() const
  {
    if(_currentTab) {
      return _currentTab->referenceCount();
    } else {
      return 0;
    }
  }

  void PropertyProxy::removeTab(uint tab, PropertyItem * item)
  {
    if(_currentCategory) {
      ExpandTabWidget * w;
      if(isActive()) {
        w=static_cast<ExpandTabWidget *>(PropertyEditor::instance()->widget(_currentCategory->caption()));
      } else {
        w=nullptr;
      }
      _currentCategory->removeTab(tab, item, w);
      if(_currentCategory->isEmpty()) {
        _categories[_currentCategory->id()]=nullptr;
        if(isActive()) {
          PropertyEditor::instance()->removeCategory(_currentCategory->caption());
        }
        delete _currentCategory;
        _currentCategory=nullptr;
      }
    }
  }

} // namespace QGpGuiTools
