/***************************************************************************
**
**  This file is part of QGpGuiTools.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-07-03
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "DoubleSpinBox.h"

namespace QGpGuiTools {

/*!
  \class DoubleSpinBox DoubleSpinBox.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
DoubleSpinBox::DoubleSpinBox(QWidget * parent)
    : QDoubleSpinBox(parent)
{
  TRACE;
  switch(rand()%6) {
  default: // More chance to get this one
    setToolTip(tr("Need more precision? Press SHIFT while rolling wheel button."));
    break;
  case 1:
    setToolTip(tr("Need a smaller step? Press ALT while rolling wheel button."));
    break;
  case 2:
    setToolTip(tr("Need a temporary bigger step? Press CTRL while rolling wheel button."));
    break;
  }
  _originalValue=0.0;
}

/*!
  Keeps track of the original value:
  \li if untouched, return the full precision value
  \li when increasing precision, do not loose the original digits
*/
void DoubleSpinBox::setValue(double val)
{
  _originalValue=val;
  QDoubleSpinBox::setValue(val);
}

/*!
  Returns true if the current value is the same as the original one, taking
  the display precision into account.
*/
bool DoubleSpinBox::touched() const
{
  return fabs((QDoubleSpinBox::value()-_originalValue)*pow(10.0, decimals()))>=1.0;
}

/*!
  Return the current value with its full precision if it is untouched
*/
double DoubleSpinBox::value() const
{
  if(touched()) {
    return QDoubleSpinBox::value();
  } else {
    return _originalValue;
  }
}

/*!
  Let user change the precision and step (alt and shift, respectively)
*/
void DoubleSpinBox::wheelEvent(QWheelEvent *event)
{
  TRACE;
  if(event->modifiers() & Qt::AltModifier) {
    if(event->angleDelta().x()>0) {  // Noticed that x is required when ALT is pressed
      setSingleStep(singleStep()*10.0);
    } else {
      if(singleStep()>=pow(0.1,decimals()-1)) {
        setSingleStep(singleStep()*0.1);
      }
    }
    showToolTip(tr("Step=%1").arg(singleStep()));
    event->accept();
  } else if(event->modifiers() & Qt::ShiftModifier) {
    if(event->angleDelta().y()>0) {
      int n=decimals();
      double f=pow(10.0, n);
      double d=QDoubleSpinBox::value()-round(_originalValue*f)/f;
      setDecimals(n+1);
      QDoubleSpinBox::setValue(_originalValue+d);
    } else {
      // Rounding is handled by QDoubleSpinBox
      setDecimals(decimals()-1);
    }
    showToolTip(tr("Decimals=%1").arg(decimals()));
    event->accept();
  } else {
    QDoubleSpinBox::wheelEvent(event);
  }
}

void DoubleSpinBox::showToolTip(const QString& text)
{
  TRACE;
  QToolTip::showText(mapToGlobal(QPoint()), text, this);
}

} // namespace QGpGuiTools
