/***************************************************************************
**
**  This file is part of QGpGuiMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-03-08
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QGpGuiMath.h>

#include "ImageScalingPoint.h"
#include "ImageScalingPointModel.h"
#include "ImageScalingPointDelegate.h"
#include "LayerPicker.h"
#include "ui_ImageScalingPoint.h"

namespace QGpGuiMath {

  /*!
    \class ImageScalingPoint ImageScalingPoint.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  ImageScalingPoint::ImageScalingPoint(QWidget * parent)
      : Dialog(parent),
        _ui(new Ui::ImageScalingPoint),
        _picker(new LayerPicker(this))
  {
    TRACE;
    _ui->setupUi(this);

    _ui->pointTable->setModel(new ImageScalingPointModel(this));
    ImageScalingPointDelegate * delegate=new ImageScalingPointDelegate(this);
    _ui->pointTable->setItemDelegate(delegate);
    connect(delegate, SIGNAL(dataChanged(const QModelIndex &, StringDiff)),
            this, SLOT(setScaling()));
    _ui->pointTable->setBackgroundMessage(tr("First click on 'Add' to pick reference points on the image.\n"
                                             "Press 'ESC' or click anywhere outside to abort.\n"
                                             "Zoom and browse with the wheel button and key modifiers (CTRL and ALT).\n"
                                             "You can import their real coordinates from a file. Names must match."));

    _ui->pointTable->setSelectionMode(QAbstractItemView::SingleSelection);
    _ui->pointTable->horizontalHeader()->setSectionResizeMode(3, QHeaderView::Stretch);
    _ui->pointTable->horizontalHeader()->setSectionResizeMode(4, QHeaderView::Stretch);

    connect(_picker, SIGNAL(picked(const Point2D&)), this, SLOT(add(const Point2D&)));
    connect(_picker, SIGNAL(finished()), this, SLOT(endPick()));

    Settings::columnWidth(_ui->pointTable, "ImageScalingPoint");
    Settings::getWidget(this);
    _layer=nullptr;
  }

  ImageScalingPoint::~ImageScalingPoint()
  {
    TRACE;
    Settings::setColumnWidth(_ui->pointTable, "ImageScalingPoint");
    Settings::setWidget(this);
  }

  void ImageScalingPoint::setLayer(ImageLayer * layer)
  {
    TRACE;
    _references=layer->references();
    model()->setReferenceList(&_references);
    setScaling();
    _layer=layer;
    _picker->addLayer(layer);
  }

  ImageScalingPointModel * ImageScalingPoint::model() const
  {
    return static_cast<ImageScalingPointModel *>(_ui->pointTable->model());
  }

  /*!
    Slot connected in ImageLayerProperties when creating this object.
    It is in charge of effectively adding the picked points.
  */
  void ImageScalingPoint::add(const Point2D& p)
  {
    TRACE;
    ImageLayer * layer=qobject_cast<ImageLayer *>(_picker->layers().first());
    if(layer) {
      model()->add(layer->r2i(p));
    }
  }

  void ImageScalingPoint::on_add_clicked()
  {
    TRACE;
    _picker->begin(window(),  QCursor(QPixmap(":scalecursor.png"), 4 , 4));
    _ui->add->setChecked(true);
  }

  void ImageScalingPoint::endPick()
  {
    TRACE;
    _picker->end(window());
    _ui->add->setChecked(false);
  }

  void ImageScalingPoint::on_remove_clicked()
  {
    TRACE;
    model()->remove(_ui->pointTable->currentIndex().row());
    setScaling();
  }

  void ImageScalingPoint::on_importCoordinates_clicked()
  {
    TRACE;

    CoordinateFileWidget cr;
    if(cr.read()) {
      QList<NamedPoint> points=cr.points();
      QMap<QString, Point> pointMap;
      for(QList<NamedPoint>::iterator it=points.begin(); it!=points.end(); it++) {
        pointMap.insert(it->name(), *it);
      }
      _utmZone=cr.utmZone();
      int n=_references.count();
      for(int i=0; i<n; i++) {
        QMap<QString, Point>::iterator it=pointMap.find(_references.at(i).name());
        if(it!=pointMap.end()) {
          model()->setReal(i, it.value());
        }
      }
    }
    setScaling();
  }

  void ImageScalingPoint::on_exportCoordinates_clicked()
  {
    TRACE;
    int n=_references.count();
    QList<NamedPoint> points;
    for(int i=0; i<n; i++) {
      points.append(NamedPoint(_references.at(i).name(), _references.at(i).real()));
    }
    CoordinateFileWidget::write(points, _utmZone);
  }

  void ImageScalingPoint::setScaling()
  {
    TRACE;
    if(_references.count()>1 && _layer) {
      Point2D o, s;
      _layer->scaling(_references, o, s);
      QString str=tr("Origin: %1,%2 ; Scale %3,%4")
                  .arg(o.x(), 0, 'f')
                  .arg(o.y(), 0, 'f')
                  .arg(s.x(), 0, 'f')
                  .arg(s.y(), 0, 'f');
      if(fabs((fabs(s.x())-fabs(s.y()))/s.x())>0.1) {
        str+=tr("\nX and Y scales differ: for maps, make sure that Y axis is parallel to north.");
      }
      _ui->scale->setText(str);
    } else {
      _ui->scale->setText(tr("Not enough point to calculate scale, pick or add new points."));
    }
  }

} // namespace QGpGuiMath
