/***************************************************************************
**
**  This file is part of QGpCoreStat.
**
**  QGpCoreStat is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreStat is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-12-08
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "MultivariateHistogram.h"
#include "GaussianMixtureDistribution.h"

namespace QGpCoreStat {

  /*!
    \class MultivariateHistogram MultivariateHistogram.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  MultivariateHistogram::MultivariateHistogram(int dimensionCount)
    : IncreaseStorage(64),
      _newBucket(dimensionCount),
      _bucket(dimensionCount, nullptr)
  {
    ASSERT(dimensionCount>=1);
    _samplings=new SamplingParameters[_bucket.count()];
    _periodic=new bool[_bucket.count()];
    _data=static_cast<int*>(allocateVector(sizeof(int)*(_bucket.count()+1)));
    _bucketCount2=1;
    _normalizationFactor=1.0;
  }

  /*!
    Description of destructor still missing
  */
  MultivariateHistogram::~MultivariateHistogram()
  {
    free(_data);
  }

  void MultivariateHistogram::setSampling(int dim, const SamplingParameters& s, bool periodic)
  {
    _samplings[dim]=s;
    _samplings[dim].linearize();
    _periodic[dim]=periodic;
  }

  void MultivariateHistogram::reallocate()
  {
    _data=static_cast<int*>(reallocateVector(_data, sizeof(int)*(_bucket.count()+1)));
  }

  void MultivariateHistogram::addSample(const Vector<double>& x)
  {
    for(int i=_bucket.count()-1; i>=0; i--) {
      _newBucket[i]=_samplings[i].index(x[i]);
    }
    bool matched=false;
    int index=indexAfter(_newBucket, matched);
    if(matched) {
      bucketValueAt(index)++;
    } else {
      add();
      int n=_bucket.count()+1;
      memmove(_data+n*(index+1), _data+n*index,  sizeof(int)*n*(size()-1-index));
      memcpy(_data+n*index, _newBucket.values(), sizeof(int)*n);
      bucketValueAt(index)=1;
      if(_bucketCount2<=size()) {
        _bucketCount2=_bucketCount2 << 1;
      }
    }
  }

  void MultivariateHistogram::printDebug()
  {
    printf("==========\n");
    for(int ib=0; ib<size(); ib++) {
      printf("%5i:", ib);
      const Vector<int>& b=bucketAt(ib);
      for(int id=0; id<b.count(); id++) {
        printf(" %5i", b[id]);
      }
      printf(" %5i\n", bucketValueAt(ib));
    }
  }

  /*!
    Returns the index right after \a bucket. Returned index can range from 0 to size().
    If \a bucket is perfectly matched, \a matched is set to true.
  */
  int MultivariateHistogram::indexAfter(const Vector<int>& bucket, bool& matched) const
  {
    int n=size();
    int i=_bucketCount2-1;
    int step2=_bucketCount2 >> 1;
    while(step2>0) {
      if(i>=n) {
        i-=step2;
      } else {
        const Vector<int>& bi=bucketAt(i);
        if(bucket<bi) {
          i-=step2;
        } else if(bucket==bi) {
          matched=true;
          return i;
        } else {
          i+=step2;
        }
      }
      step2=step2 >> 1;
    }
    if(i<n) {
      const Vector<int>& bi=bucketAt(i);
      if(bucket>bi) {
        matched=false;
        return i+1;
      } else if(bucket==bi) {
        matched=true;
        return i;
      }
    }
    matched=false;
    return i;
  }

  void MultivariateHistogram::normalize()
  {
    _normalizationFactor=0.0;
    for(int i=size()-1; i>=0; i--) {
      _normalizationFactor+=bucketValueAt(i);
    }
    double bucketHyperVolume=1.0;
    for(int i=_bucket.count()-1; i>=0; i--) {
      bucketHyperVolume*=_samplings[i].step();
    }
    _normalizationFactor=1.0/(_normalizationFactor*bucketHyperVolume);
  }

  double MultivariateHistogram::misfit(const GaussianMixtureDistribution& f) const
  {
    double m=0.0, d;
    PrivateVector<double> pos(_bucket.count());
    for(int i=size()-1; i>=0; i--) {
      bucketPosition(i, pos);
      d=normalizedBucketValueAt(i)-f.value(pos);
      m+=d*d;
    }
    return m;
  }

  void MultivariateHistogram::removeSmallBucketValues(double minimum)
  {
    VectorList<int> toRemove;
    for(int i=0; i<size(); i++) {
      if(normalizedBucketValueAt(i)<=minimum) {
        toRemove.append(i);
      }
    }
    if(!toRemove.isEmpty()) {
      int bucketSize=_bucket.count()+1;
      int removeCount=toRemove.count()-1;
      int index, moveSize;
      int offset=0;
      for(int i=0; i<removeCount; i++) {
        index=toRemove.at(i);
        moveSize=toRemove.at(i+1)-index-1;
        memmove(_data+bucketSize*(index-offset), _data+bucketSize*(index+1),
                sizeof(int)*bucketSize*moveSize);
        offset++;
      }
      index=toRemove.last();
      moveSize=size()-index-1;
      memmove(_data+bucketSize*(index-offset), _data+bucketSize*(index+1),
              sizeof(int)*bucketSize*moveSize);
      offset++;
      removeLast(offset);
    }
  }

  MultivariateStatistics * MultivariateHistogram::statistics() const
  {
    int bucketSize=_bucket.count();
    MultivariateStatistics * stat=new MultivariateStatistics(bucketSize);
    PrivateVector<double> pos(bucketSize);
    for(int ib=size()-1; ib>=0; ib--) {
      const Vector<int>& bucket=bucketAt(ib);
      for(int i=0; i<bucketSize; i++) {
        pos[i]=_samplings[i].value(bucket[i]);
      }
      stat->add(pos, bucket[bucketSize]);
    }
    return stat;
  }

  void MultivariateHistogram::effectiveLimits(Vector<double>& min, Vector<double>& max) const
  {
    min.setValue(std::numeric_limits<double>::infinity());
    max.setValue(-std::numeric_limits<double>::infinity());
    for(int ib=size()-1; ib>=0; ib--) {
      const Vector<int>& b=bucketAt(ib);
      for(int id=_bucket.count()-1; id>=0; id--) {
        if(b[id]<min[id]) {
          min[id]=b[id];
        }
        if(b[id]>max[id]) {
          max[id]=b[id];
        }
      }
    }
    for(int i=_bucket.count()-1; i>=0; i--) {
      min[i]=_samplings[i].value(min[i]);
      max[i]=_samplings[i].value(max[i]);
    }
  }

  int MultivariateHistogram::sampleCount() const
  {
    int n=0;
    for(int ib=size()-1; ib>=0; ib--) {
      n+=bucketValueAt(ib);
    }
    return n;
  }

  void MultivariateHistogram::smooth()
  {
    int n=0;
    for(int ib=size()-1; ib>=0; ib--) {
      n+=bucketValueAt(ib);
    }
  }

} // namespace QGpCoreStat

