/***************************************************************************
**
**  This file is part of QGpCoreStat.
**
**  QGpCoreStat is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreStat is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-06-11
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "GaussianMixtureLikelihood.h"

namespace QGpCoreStat {

  /*!
    \class GaussianMixtureLikelihood GaussianMixtureLikelihood.h
    \brief The likelihood of a Gaussian mixture distribution

    Computes the likelihood of a Gaussian mixture distribution
  */

  /*!
    Description of constructor still missing
  */
  GaussianMixtureLikelihood::GaussianMixtureLikelihood(const GaussianMixtureParameters * p,
                                                       int dimensionCount)
    : AbstractForward()
  {
    TRACE;
    _dimensionCount=dimensionCount;
    _parameters=p;
    _histogram=nullptr;
    _distribution=nullptr;
  }

  /*!
    Description of constructor still missing
  */
  GaussianMixtureLikelihood::GaussianMixtureLikelihood(const GaussianMixtureLikelihood& o)
    : AbstractForward()
  {
    TRACE;
    _dimensionCount=o._dimensionCount;
    _parameters=o._parameters;
    _histogram=o._histogram;
    if(o._distribution) {
      _distribution=new GaussianMixtureDistribution(*o._distribution);
    } else {
      _distribution=nullptr;
    }
  }

  /*!
    Description of destructor still missing
  */
  GaussianMixtureLikelihood::~GaussianMixtureLikelihood()
  {
    TRACE;
    delete _distribution;
  }

  AbstractForward * GaussianMixtureLikelihood::clone() const
  {
    TRACE;
    GaussianMixtureLikelihood * f=new GaussianMixtureLikelihood(*this);
    if(f->setInversionParameters()) {
      return f;
    } else {
      return nullptr;
    }
  }

  /*!
    Initialize parameters for optimization
  */
  bool GaussianMixtureLikelihood::setInversionParameters()
  {
    TRACE;
    if(_histogram->isEmpty()) {
      App::log(tr("Empty multivariate histogram, aborting inversion\n"));
      return false;
    }
    RealSpace& ps=parameterSpace();
    ps.clearParameters();
    QString meanName("m%1"), stddevName("s%1");
    Parameter * m1, *m2;
    double meanSpacingFactor=_parameters->meanSpacingFactor();
    PrivateVector<double> minima(_dimensionCount), maxima(_dimensionCount);
    _histogram->effectiveLimits(minima, maxima);
    int modeCount=_parameters->maximumModeCount(minima[0], maxima[0]);
    if(_parameters->modeCount()<modeCount) {
      modeCount=_parameters->modeCount();
    }
    if(modeCount<1) {
      App::log(tr("Maximum number of modes below 1, aborting inversion\n"));
      return false;
    }
    delete _distribution;
    _distribution=new GaussianMixtureDistribution(modeCount, _dimensionCount);
    for(int id=0; id<_dimensionCount; id++) {
      m2=nullptr;
      for(int im=0; im<_distribution->modeCount(); im++) {
        m1=m2;
        m2=ps.addParameter(meanName.arg(im), "", minima[id], maxima[id],
                           ParameterGrid::Linear, _histogram->sampling(id).step());
        if(!m2) {
          return false;
        }
        if(id==0 && m1) {
          if(m1->minimum()>0.0) {
            ps.addCondition(new SimpleCondition(m2, SimpleCondition::GreaterThan, meanSpacingFactor, m1, 0.0));
          } else {
            ps.addCondition(new SimpleCondition(m2, SimpleCondition::LessThan, meanSpacingFactor, m1, 0.0));
          }
        }
        double ref=_histogram->sampling(id).step();
        ps.addParameter(stddevName.arg(im), "", 0.1*ref, 50.0*ref, ParameterGrid::Log, 0.0);
      }
    }
    setWeightParameters();
    return true;
  }

  void GaussianMixtureLikelihood::setWeightParameters()
  {
    RealSpace& ps=parameterSpace();
    QString weightName("w%1");
    Parameter * w;
    SumCondition * weightCond=new SumCondition(SumCondition::LessThan, 1.0);
    for(int i=1; i<_distribution->modeCount(); i++) {
      w=ps.addParameter(weightName.arg(i), "", 0.0, 1.0, ParameterGrid::Linear, 0.0);
      weightCond->addParameter(w);
    }
    ps.addCondition(weightCond);
  }

  void GaussianMixtureLikelihood::parametersToModes()
  {
    TRACE;
    int pindex;
    RealSpace& ps=parameterSpace();
    for(int im=0; im<_distribution->modeCount(); im++) {
      for(int id=0; id<_dimensionCount; id++) {
        pindex=2*(im+id*_distribution->modeCount());
        _distribution->setMode(im, id, ps.parameter(pindex)->realValue(),
                                       ps.parameter(pindex+1)->realValue());
      }
      _distribution->setFactor(im);
    }
    parametersToWeights(2*_distribution->modeCount()*_dimensionCount);
  }

  void GaussianMixtureLikelihood::parametersToWeights(int firstWeightIndex)
  {
    TRACE;
    double w;
    double w0=1.0;
    firstWeightIndex--;
    for(int i=1; i<_distribution->modeCount(); i++) {
      w=parameterSpace().parameter(firstWeightIndex+i)->realValue();
      _distribution->setWeight(i, w);
      w0-=w;
    }
    ASSERT(w0>=0);
    _distribution->setWeight(0, w0);
  }

  double GaussianMixtureLikelihood::misfit(bool&)
  {
    TRACE;
    parametersToModes();
    return _histogram->misfit(*_distribution);
  }

  void GaussianMixtureLikelihood::setValues(const int * values)
  {
    TRACE;
    int n=parameterSpace().variableParameterCount();
    for(int i=0; i<n; i++) {
      parameterSpace().parameter(i)->setGridValue(values[i]);
    }
    parametersToModes();
  }

} // namespace QGpCoreStat

