/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-09-16
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "TwiddleFactors.h"

namespace QGpCoreMath {

  /*!
    \class TwidleFactors TwidleFactors.h
    \brief Memory optimized storage for twiddle factors

    Split twiddle factors in two sub vectors to minimize the memory usage,
    and optimize cache usage.
  */

  /*!
    Twiddle factors for FFTW must be initialized with \a amp=1 and \a arg=1/n.
  */
  TwiddleFactors::TwiddleFactors(int n, double amp, double arg)
  {
    ASSERT(n>0);
    int base=Number::nextPowerOfTwo(qCeil(sqrt(n)));
    _mask=base-1;
    _shift=0;
    for(int i=1; i<=_mask; i=i << 1, _shift++) {}

    double omega=-2.0*M_PI*arg;
    _low=new Complex[base];
    _low[0]=1.0;
    for(int k=_mask; k>=1; k--) {
      _low[k].setUnitExp(static_cast<double>(k)*omega);
    }

    int highCount=((n & ~_mask) >> _shift)+1;
    _high=new Complex[highCount];
    _high[0]=amp;
    for(int k=highCount-1; k>0; k--) {
      _high[k].setExp(amp, static_cast<double>(k*base)*omega);
    }
  }

  TwiddleFactors::~TwiddleFactors()
  {
    delete [] _low;
    delete [] _high;
  }

  void TwiddleFactors::print(int n)
  {
    printf("  Base  %i\n"
           "  Shift %i\n"
           "  Mask  %i\n"
           "  Low count  %i\n"
           "  High count %i\n", _mask+1, _shift, _mask, _mask+1, ((n & ~_mask) >> _shift)+1);
  }

  bool TwiddleFactors::testAll(int minN, int maxN)
  {
    for(int i=minN; i<=maxN; i++) {
      printf("Testing %i...\n", i);
      if(!test(i)) {
        return false;
      }
    }
    return true;
  }

  bool TwiddleFactors::test(int n)
  {
    TwiddleFactors f(n, 1.0, 1.0/static_cast<double>(n));
    f.print(n);
    double omega=-2.0*M_PI/static_cast<double>(n);
    Complex c1, c2;
    for(int k=0; k<=n; k++) {
      c1.setUnitExp(static_cast<double>(k)*omega);
      c2=f.value(k);
      if(fabs(c1.re()-c2.re())>1e-15 ||
         fabs(c1.im()-c2.im())>1e-15) {
        App::log(tr("Test failed for n=%1 at k=%2:\n"
                    "   %3 instead of %4\n").arg(n).arg(k).arg(c2.toString()).arg(c1.toString()));
        return false;
      }
    }
    return true;
  }

} // namespace QGpCoreMath

