/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-07-01
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>
#include <limits>

#include "Statistics.h"

namespace QGpCoreMath {

  /*!
    \class Statistics Statistics.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  void Statistics::reset()
  {
    _sum=0.0;
    _sum2=0.0;
    _sumWeight=0.0;
    _sumWeight2=0.0;
  }

  /*!
    
  */
  void Statistics::add(double val, double weight)
  {
    double c=val*weight;
    _sum+=c;
    _sum2+=c*val;
    _sumWeight+=weight;
    _sumWeight2+=weight*weight;
  }

  double Statistics::mean() const
  {
    if(_sumWeight<=0.0) {
      return std::numeric_limits<double>::quiet_NaN();
    } else {
      return _sum/_sumWeight;
    }
  }

  /*!
    Unbiased weighted sample variance
    https://en.wikipedia.org/wiki/Weighted_arithmetic_mean#Related_concepts
    https://en.wikipedia.org/wiki/Variance#Unbiased_sample_variance

    If all N weights are equal to 1, we have
      sigma^2=(_sum2-_sum^2/N)/(N-1)
  */
  double Statistics::variance() const
  {
    if(_sumWeight<=0.0) {
      return std::numeric_limits<double>::quiet_NaN();
    } else {
      double invSumWeight=1.0/_sumWeight;
      double unbiasFactor=_sumWeight-_sumWeight2*invSumWeight;
      if(unbiasFactor>0.0) {
        return (_sum2-_sum*_sum*invSumWeight)/unbiasFactor;
      } else {
        return 0.0;
      }
    }
  }

  /*!
    Unbiased weighted sample variance for a defined average (median, mode,...)
  */
  double Statistics::variance(double average) const
  {
    if(_sumWeight<=0.0) {
      return std::numeric_limits<double>::quiet_NaN();
    } else {
      double unbiasFactor=_sumWeight-_sumWeight2/_sumWeight;
      if(unbiasFactor>0.0) {
        return (_sum2+(average*_sumWeight-2.0*_sum)*average)/unbiasFactor;
      } else {
        return 0.0;
      }
    }
  }

  double Statistics::stddev() const
  {
    double s=variance();
    if(s<0.0) s=0.0;
    return sqrt(s);
  }

  double Statistics::stddev(double average) const
  {
    double s=variance(average);
    if(s<0.0) s=0.0;
    return sqrt(s);
  }

} // namespace QGpCoreMath
