/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-06-15
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef PRIVATEVECTOR_H
#define PRIVATEVECTOR_H

#include <QGpCoreTools.h>

#include "Vector.h"
#include "QGpCoreMathDLLExport.h"

namespace QGpCoreMath {

  template <typename T> class QGPCOREMATH_EXPORT PrivateVector : public Vector<T>
  {
  public:
    inline PrivateVector(int count);
    inline PrivateVector(int count, const T& value);
    inline PrivateVector(int count, T * values);
    inline PrivateVector(const Vector<T>& o);
    inline PrivateVector(const PrivateVector<T>& o);
    inline PrivateVector(const VectorList<T>& o);
    inline PrivateVector(const Vector<T>& o, int firstIndex, int lastIndex);
    inline PrivateVector(const PrivateVector<T>& o, int firstIndex, int lastIndex);
    inline PrivateVector(const Vector<T>& o, int firstIndex, int lastIndex, double factor);
    inline PrivateVector(const PrivateVector<T>& o, int firstIndex, int lastIndex, double factor);
    inline ~PrivateVector();

    inline void operator=(const Vector<T>& o);
    inline void operator=(const PrivateVector<T>& o);

    inline void setValues(T * values);
    inline void setValues(const T& v1);
    inline void setValues(const T& v1, const T& v2);
    inline void setValues(const T& v1, const T& v2, const T& v3);
    inline void setValues(const T& v1, const T& v2, const T& v3, const T& v4);

    inline void resize(int count);
  };

  template <typename T>
  inline PrivateVector<T>::PrivateVector(int count)
    : Vector<T>(count, static_cast<T *>(malloc(count*sizeof(T))))
  {
  }

  template <typename T>
  inline PrivateVector<T>::PrivateVector(int count, const T& value)
    : Vector<T>(count, static_cast<T *>(malloc(count*sizeof(T))))
  {
    for(int i=0; i<count; i++) {
      Vector<T>::at(i)=value;
    }
  }

  template <typename T>
  inline PrivateVector<T>::PrivateVector(int count, T * values)
    : Vector<T>(count, static_cast<T *>(malloc(count*sizeof(T))))
  {
    memcpy(Vector<T>::values(), values, count*sizeof(T));
  }

  template <typename T>
  inline void PrivateVector<T>::setValues(const T& v1)
  {
    Vector<T>::at(0)=v1;
  }

  template <typename T>
  inline void PrivateVector<T>::setValues(const T& v1, const T& v2)
  {
    Vector<T>::at(0)=v1;
    Vector<T>::at(1)=v2;
  }

  template <typename T>
  inline void PrivateVector<T>::setValues(const T& v1, const T& v2, const T& v3)
  {
    Vector<T>::at(0)=v1;
    Vector<T>::at(1)=v2;
    Vector<T>::at(2)=v3;
  }

  template <typename T>
  inline void PrivateVector<T>::setValues(const T& v1, const T& v2, const T& v3, const T& v4)
  {
    Vector<T>::at(0)=v1;
    Vector<T>::at(1)=v2;
    Vector<T>::at(2)=v3;
    Vector<T>::at(3)=v4;
  }

  template <typename T>
  inline PrivateVector<T>::PrivateVector(const Vector<T>& o)
    : Vector<T>(o.count(), static_cast<T *>(malloc(o.count()*sizeof(T))))
  {
    memcpy(Vector<T>::values(), o.values(), Vector<T>::count()*sizeof(T));
  }

  template <typename T>
  inline PrivateVector<T>::PrivateVector(const VectorList<T>& o)
    : Vector<T>(o.count(), static_cast<T *>(malloc(o.count()*sizeof(T))))
  {
    memcpy(Vector<T>::values(), o.data(), Vector<T>::count()*sizeof(T));
  }

  template <typename T>
  inline PrivateVector<T>::PrivateVector(const PrivateVector<T>& o)
    : Vector<T>(o.count(), static_cast<T *>(malloc(o.count()*sizeof(T))))
  {
    memcpy(Vector<T>::values(), o.values(), Vector<T>::count()*sizeof(T));
  }

  template <typename T>
  inline PrivateVector<T>::PrivateVector(const Vector<T>& o, int firstIndex, int lastIndex)
    : Vector<T>(lastIndex-firstIndex+1, nullptr)
  {
    Vector<T>::setValues(static_cast<T *>(malloc(Vector<T>::count()*sizeof(T))));
    for(int i=firstIndex; i<=lastIndex; i++) {
      Vector<T>::at(i-firstIndex)=o.at(i);
    }
  }

  template <typename T>
  inline PrivateVector<T>::PrivateVector(const PrivateVector<T>& o, int firstIndex, int lastIndex)
    : Vector<T>(lastIndex-firstIndex+1, nullptr)
  {
    Vector<T>::setValues(static_cast<T *>(malloc(Vector<T>::count()*sizeof(T))));
    for(int i=firstIndex; i<=lastIndex; i++) {
      Vector<T>::at(i-firstIndex)=o.at(i);
    }
  }

  template <typename T>
  inline PrivateVector<T>::PrivateVector(const Vector<T>& o, int firstIndex, int lastIndex, double factor)
    : Vector<T>(lastIndex-firstIndex+1, nullptr)
  {
    Vector<T>::setValues(static_cast<T *>(malloc(Vector<T>::count()*sizeof(T))));
    for(int i=firstIndex; i<=lastIndex; i++) {
      Vector<T>::at(i-firstIndex)=o.at(i)*factor;
    }
  }

  template <typename T>
  inline PrivateVector<T>::PrivateVector(const PrivateVector<T>& o, int firstIndex, int lastIndex, double factor)
    : Vector<T>(lastIndex-firstIndex+1, nullptr)
  {
    Vector<T>::setValues(static_cast<T *>(malloc(Vector<T>::count()*sizeof(T))));
    for(int i=firstIndex; i<=lastIndex; i++) {
      Vector<T>::at(i-firstIndex)=o.at(i)*factor;
    }
  }

  template <typename T>
  inline PrivateVector<T>::~PrivateVector()
  {
    free(Vector<T>::values());
  }

  template <typename T>
  inline void PrivateVector<T>::setValues(T * values)
  {
    free(Vector<T>::values());
    Vector<T>::setValues(values);
  }

  template <typename T>
  inline void PrivateVector<T>::resize(int count)
  {
    if(count!=Vector<T>::count()) {
      Vector<T>::setValues(static_cast<T *>(realloc(Vector<T>::values(), count)));
      Vector<T>::resize(count);
    }
  }

  template <typename T>
  inline void PrivateVector<T>::operator=(const Vector<T>& o)
  {
    resize(o.count());
    memcpy(Vector<T>::values(), o.Vector<T>::values(), Vector<T>::count()*sizeof(T));
  }

  template <typename T>
  inline void PrivateVector<T>::operator=(const PrivateVector<T>& o)
  {
    resize(o.count());
    memcpy(Vector<T>::values(), o.Vector<T>::values(), Vector<T>::count()*sizeof(T));
  }

} // namespace QGpCoreMath

#endif // PRIVATEVECTOR_H

