/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-03-21
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Polynomial.h"

namespace QGpCoreMath {

  /*!
    \class Polynomial Polynomial.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  Polynomial::Polynomial(int degree)
    : _coefficients(degree+1, 0.0)
  {
    ASSERT(degree>=0);
  }

  Polynomial::Polynomial(double c0, double c1)
    : _coefficients(2)
  {
    _coefficients[0]=c0;
    _coefficients[1]=c1;
  }

  Polynomial::Polynomial(double c0, double c1, double c2)
    : _coefficients(3)
  {
    _coefficients[0]=c0;
    _coefficients[1]=c1;
    _coefficients[2]=c2;
  }

  Polynomial::Polynomial(double c0, double c1, double c2, double c3)
    : _coefficients(4)
  {
    _coefficients[0]=c0;
    _coefficients[1]=c1;
    _coefficients[2]=c2;
    _coefficients[3]=c3;
  }

  void Polynomial::setDegree(int newDegree)
  {
    ASSERT(newDegree>=0);
    if(newDegree!=degree()) {
      _coefficients.resize(newDegree+1);
      // Check if it necessary
      for(int i=degree()+1; i<newDegree; i++) {
        _coefficients[i]=0.0;
      }
    }
  }

  int Polynomial::effectiveDegree() const
  {
    int d=degree();
    while(d>=0 && _coefficients.at(d)==0.0) {
      d--;
    }
    if(d<0) {
      return 0;
    } else {
      return d;
    }
  }

  double Polynomial::value(double x) const
  {
    double sum=0.0;
    int d=degree();
    double xn=1.0;
    for(int i=0; i<=d; i++) {
      sum+=_coefficients.at(i)*xn;
      xn*=x;
    }
    return sum;
  }

  VectorList<Complex> Polynomial::allRoots() const
  {
    int d=effectiveDegree();
    DoubleMatrix companion(d);
    companion.zero();
    double inverseLeadingCoefficient=-1.0/_coefficients.at(d);
    d--;
    for(int i=1; i<=d; i++) {
      companion.at(i, i-1)=1.0;
      companion.at(i-1, d)=_coefficients.at(i-1)*inverseLeadingCoefficient;
    }
    companion.at(d, d)=_coefficients.at(d)*inverseLeadingCoefficient;
    return companion.eigenValues();
  }

  VectorList<double> Polynomial::realRoots(double roundPrecision, double minImaginaryRatio) const
  {
    VectorList<Complex> complexRoots=allRoots();
    VectorList<double> realRoots;
    int n=complexRoots.count();
    for(int i=0; i<n; i++) {
      const Complex& r=complexRoots.at(i);
      if(fabs(r.im()/r.re())<minImaginaryRatio) {
        double v=r.re();
        if(roundPrecision>0.0) {
          v=Number::round(v, roundPrecision);
        }
        realRoots.append(v);
      }
    }
    std::sort(realRoots.begin(), realRoots.end());
    if(roundPrecision>0.0) { // If no precision limit specified, multiple roots
                             // are numerically almost never exactly equal.
      unique(realRoots);
    }
    return realRoots;
  }

  Polynomial Polynomial::operator*(const Polynomial& o) const
  {
    int d1=degree();
    int d2=o.degree();
    Polynomial r(d1+d2);
    for(int i=0; i<=d1; i++) {
      for(int j=0; j<=d2; j++) {
        r._coefficients[i+j]+=_coefficients.at(i)*o._coefficients.at(j);
      }
    }
    return r;
  }

  Polynomial Polynomial::operator+(const Polynomial& o) const
  {
    int d;
    int d1=degree();
    int d2=o.degree();
    if(d1<d2) {
      d=d2;
    } else {
      d=d1;
    }
    Polynomial r(d);
    for(int i=0; i<=d1; i++) {
      r._coefficients[i]=_coefficients.at(i);
    }
    for(int i=0; i<=d2; i++) {
      r._coefficients[i]+=o._coefficients.at(i);
    }
    return r;
  }

  Polynomial Polynomial::operator-(const Polynomial& o) const
  {
    int d;
    int d1=degree();
    int d2=o.degree();
    if(d1<d2) {
      d=d2;
    } else {
      d=d1;
    }
    Polynomial r(d);
    for(int i=0; i<=d1; i++) {
      r._coefficients[i]=_coefficients.at(i);
    }
    for(int i=0; i<=d2; i++) {
      r._coefficients[i]-=o._coefficients.at(i);
    }
    return r;
  }

  void Polynomial::operator+=(const Polynomial& o)
  {
    int d=o.degree();
    if(degree()<d) {
      setDegree(d);
    }
    for(int i=0; i<=d; i++) {
      _coefficients[i]+=o._coefficients.at(i);
    }
  }

  void Polynomial::operator-=(const Polynomial& o)
  {
    int d=o.degree();
    if(degree()<d) {
      setDegree(d);
    }
    for(int i=0; i<=d; i++) {
      _coefficients[i]-=o._coefficients.at(i);
    }
  }

  Polynomial Polynomial::operator*(double f) const
  {
    Polynomial r(*this);
    r*=f;
    return r;
  }

  Polynomial Polynomial::operator/(double f) const
  {
    Polynomial r(*this);
    r/=f;
    return r;
  }

  void Polynomial::operator*=(double f)
  {
    int d=degree();
    for(int i=0; i<=d; i++) {
      _coefficients[i]*=f;
    }
  }

  void Polynomial::operator/=(double f)
  {
    f=1.0/f;
    int d=degree();
    for(int i=0; i<=d; i++) {
      _coefficients[i]*=f;
    }
  }

  void Polynomial::derivate()
  {
    int d=degree();
    for(int i=1; i<=d; i++) {
      _coefficients[i-1]=i*_coefficients.at(i);
    }
    _coefficients[d]=0.0;
  }

  QString Polynomial::toString() const
  {
    const QString formatExpN("%1x^%2");
    const QString formatCoef1ExpN("x^%2");
    const QString formatExp1("%1x");
    const QString formatCoef1Exp1("x");
    const QString formatExp0("%1");
    QString r;
    int d=effectiveDegree();
    if(d>=2) {
      const double& c=_coefficients.at(d);
      if(c==1.0) {
        r=formatCoef1ExpN.arg(d);
      } else if(c==-1.0) {
        r="-"+formatCoef1ExpN.arg(d);
      } else {
        r=formatExpN.arg(c).arg(d);
      }
    }
    for(int i=d-1; i>1; i--) {
      const double& c=_coefficients.at(i);
      if(c>0.0) {
        if(c==1.0) {
          r+="+"+formatCoef1ExpN.arg(i);
        } else {
          r+="+"+formatExpN.arg(c).arg(i);
        }
      } else if(c<0.0) {
        if(c==-1.0) {
          r+="-"+formatCoef1ExpN.arg(i);
        } else {
          r+=formatExpN.arg(c).arg(i);
        }
      }
    }
    if(d>=1) {
      const double& c=_coefficients.at(1);
      if(c<0.0) {
        if(c==-1.0) {
          r+="-"+formatCoef1Exp1;
        } else {
          r+=formatExp1.arg(c);
        }
      } else if(c>0.0) {
        if(!r.isEmpty()) {
          r+="+";
        }
        if(c==1.0) {
          r+=formatCoef1Exp1;
        } else {
          r+=formatExp1.arg(c);
        }
      }
    }
    if(d>=0) {
      const double& c=_coefficients.at(0);
      if(c>0.0 && !r.isEmpty()) {
        r+="+"+formatExp0.arg(c);
      } else if(c!=0.0){
        r+=formatExp0.arg(c);
      }
    }
    return r;
  }

  /*!
    Returns the factor between this and \a o if the two polynomials are
    equal by one factor.

    f*o=this
  */
  double Polynomial::multiple(const Polynomial& o, bool& ok) const
  {
    int d=degree();
    if(d==o.degree()) {
      double f=_coefficients[0]/o._coefficients[0];
      for(int i=1; i<=d; i++) {
        if(fabs(f*o._coefficients[i]-_coefficients[i])>1e-15) {
          ok=false;
          return 0.0;
        }
      }
      ok=true;
      return f;
    } else {
      ok=false;
      return 0.0;
    }
  }

} // namespace QGpCoreMath

