/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-10-13
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "Polygon.h"

namespace QGpCoreMath {

  /*!
    \class Polygon Polygon.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  Segment2D Polygon::segmentAt(int index) const
  {
    int n=size();
    if(index>0 && index<n) {
      return Segment2D(pointAt(index-1), pointAt(index));
    } else {
      return Segment2D(pointAt(n-1), pointAt(0));
    }
  }

  /*!
    Returns true if \a p is inside.
  */
  bool Polygon::contains(const Point2D& p) const
  {
    int n=_points.size();
    bool ok;
    int leftCount=0, rightCount=0;
    for(int i=1; i<n; i++) {
      Segment2D s=segmentAt(i);
      double x=horizontalLineIntersection(s, p.y(), ok);
      if(ok) {
        if(x<p.x()) {
          leftCount++;
        } else {
          rightCount++;
        }
      }
    }
    return (leftCount%2==1 && rightCount%2==1);
  }

  /*!
    Returns the intersection between segment \a s and an horizontal line
    passing at \a atY. If there is no intersection, \a ok is set to false.
    It is set to true if there is an intersection.
  */
  double Polygon::horizontalLineIntersection(const Segment2D& s, double atY, bool& ok) const
  {
    double dy=s.p2().y()-s.p1().y();
    if(dy==0.0) {
      ok=false;
      return 0.0;
    } else {
      double x=(s.p2().x()-s.p1().x())/dy*(atY-s.p1().y())+s.p1().x();
      ok=s.strictIncludesOnLine(Point2D(x, atY));
      return x;
    }
  }

  /*!
    Projects point \a p on polygon and move along the polygon by \a delta.
  */
  Point2D Polygon::project(const Point2D& p, double delta) const
  {
    int n=_points.size();
    double d, dmin=std::numeric_limits<double>::infinity();
    int iSeg=0;
    for(int i=0; i<n; i++) {
      Segment2D s=segmentAt(i);
      d=s.distanceTo(p);
      if(d<dmin) {
        dmin=d;
        iSeg=i;
      }
    }
    Segment2D s=segmentAt(iSeg);
    Point2D pp=s.project(p);
    if(!s.includesOnLine(pp)) {
      double d1=p.distanceTo(s.p1());
      double d2=p.distanceTo(s.p2());
      if(d1<d2) {
        pp=s.p1();
      } else {
        pp=s.p2();
      }
    }
    if(delta>0.0) {
      while(true) {
        d=pp.distanceTo(s.p2());
        if(d>=delta) {
          return s.pointAtFrom2(d-delta);
        } else {
          delta-=d;
          pp=s.p2();
          iSeg++;
          if(iSeg==n) { // several loops are possible for large delta
            iSeg=0;
          }
          s=segmentAt(iSeg);
        }
      }
    } else {
      delta=-delta;
      while(true) {
        d=pp.distanceTo(s.p1());
        if(d>=delta) {
          return s.pointAtFrom1(d-delta);
        } else {
          delta-=d;
          pp=s.p1();
          iSeg--;
          if(iSeg==0) { // several loops are possible for large delta
            iSeg=n;
          }
          s=segmentAt(iSeg);
        }
      }
    }
  }

} // namespace QGpCoreMath

