/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-05-12
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>

#include "Line2D.h"

namespace QGpCoreMath {

  /*!
    \fn Line2D::Line2D(double c)

    Construct a vertical line (parallel to y axis) crossing at (c, 0)

    line equation is a * x + b * y=\a c (with a=1 and b=0).
  */

  /*!
    \fn Line2D::Line2D(double a, double c)

    Construct any line except vertical ones.

    line equation is a * x + \a b * y=\a c (with b=1)
  */

  /*!
    Construct any line passing by point ( \a x1, \a  y1) and with azimuth \a az (counted clockwise from North).

    line equation is a * x + b * y=c
  */
  Line2D::Line2D(double x1, double y1, double az)
  {
    if(az==0.0 || az==M_PI) {
      _a=0.0;
      _b=1.0;
      _c=x1;
    } else {
      _a=1.0;
      _b=-::tan(0.5*M_PI - az);
      _c=y1+_b*x1;
    }
  }

  Line2D::Line2D(const Point2D& p1, const Point2D& p2)
  {
    ASSERT(p1!=p2);
    _a=p1.y()-p2.y();
    _b=p2.x()-p1.x();
    _c=p1.y()*p2.x()-p2.y()*p1.x();
  }

  /*!
    \fn Line2D::Line2D(const Line2D &o)

    Copy constructor
  */

  /*!
    Returns interstection between this line and line \a o. If no intersection is found, \a ok is set to false.
  */
  Point2D Line2D::intersect(const Line2D& o, bool& ok) const
  {
    TRACE;
    // _a and _b cannot be simultaneously null
    Point2D inter;
    if(_a==o._a && _b==o._b) {
      // Either no intersection or infinite noumber of intersections
      ok=false;
    } else if(_b==0.0) { // this is vertical
      inter.setX(_c/_a);
      inter.setY((o._c-o._a*inter.x())/o._b);
      ok=true;
    } else if(o._b==0.0) { // o is vertical
      inter.setX(o._c/o._a);
      inter.setY((_c-_b*inter.x())/_a);
      ok=true;
    } else {
      double invb=1.0/_b;
      double oinvb=1.0/o._b;
      inter.setX((_c*invb-o._c*oinvb)/(_a*invb-o._a*oinvb));
      inter.setY((_c-_a*inter.x())*invb);
      ok=true;
    }
    return inter;
  }

  /*!
    Forces to a perpendicular line passing by \a p
  */
  void Line2D::perpendicular(const Point2D& p)
  {
    std::swap(_a, _b);
    _b=-_b;
    parallel(p);
  }

  /*!
    Forces to a parallel line passing by \a p
  */
  void Line2D::parallel(const Point2D& p)
  {
    _c=_a*p.x()+_b*p.y();
  }

  Point2D Line2D::direction() const
  {
    Point2D d;
    if(_a>0.0) {
      d=Point2D(-_b, _a);
    } else {
      d=Point2D(_b, -_a);
    }
    d/=d.length();
    return d;
  }

  /*
  // Testing Line2D
  Point2D pa(2, 1);
  Point2D pb(0, 0);
  Line2D l(pa, pb);
  qDebug() << l.a() << l.b() << l.c();
  qDebug() << l.y(5) << l.direction().toString();
  l.parallel(Point2D(10, 1));
  qDebug() << l.a() << l.b() << l.c();
  qDebug() << l.y(5) << l.direction().toString();
  Point2D d1=l.direction();
  Line2D lp(l);
  lp.perpendicular(Point2D(10, 1));
  qDebug() << lp.a() << lp.b() << lp.c();
  qDebug() << lp.y(5) << lp.direction().toString();
  qDebug() << Angle::radiansToDegrees(d1.azimuth()) << Angle::radiansToDegrees(lp.direction().azimuth())
           << Angle::radiansToDegrees(d1.azimuth()-lp.direction().azimuth());
  bool ok=true;
  qDebug() << l.intersect(lp, ok).toString();
  Rect r(Point2D(1, 10), Point2D(9.5, 2));
  Point2D i1, i2;
  r.intersect(lp, i1, i2);
  qDebug() << i1.toString() << i2.toString();
  return 0;
  */

  bool Line2D::isOnSameSide(const Point2D& p1, const Point2D& p2)
  {
    Point2D normal(_a, _b);
    Point2D ref;  // A reference point on the line
    if(_b==0.0) {
      ref.setX(_c/_a);
      ref.setY(p1.y());
    } else {
      ref.setX(p1.x());
      ref.setY(y(p1.x()));
    }
    return ((p1-ref).scalarProduct(normal))*((p2-ref).scalarProduct(normal))>0.0;
  }

} // namespace QGpCoreMath
