/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-06-18
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "GridSearch3D.h"

namespace QGpCoreMath {

  // Get a detailled timing report of the processing
  // Run with verbosity level to 3 to get the report.
  // Configure with '-D PROCESS_TIMING'
#ifdef PROCESS_TIMING
    static double ttgb=0.0, ttlc=0.0, ttrf=0.0;
    static Mutex globalTimeMutex;
#endif

  /*!
    \class GridSearch3D GridSearch3D.h
    \brief Brief description of class still missing


  */

  /*!
    Description of constructor still missing
  */
  GridSearch3D::GridSearch3D()
    : FunctionSearch(3),
      _min(3, 0),
      _max(3, 0),
      _step(3, 0.0)
  {
    TRACE;
  }

  /*!
    Description of destructor still missing
  */
  GridSearch3D::~GridSearch3D()
  {
    TRACE;
  }

  void GridSearch3D::showProcessingTime()
  {
#ifdef PROCESS_TIMING
    App::log(3, tr("GridSearch3D: %1 %2 %3 (ms; global, local, refine)\n")
             .arg(ttgb*1e-6)
             .arg(ttlc*1e-6)
             .arg(ttrf*1e-6));
#endif
  }

  /*!
    Create a symetric gridding identical along X and Y.
  */
  void GridSearch3D::setGrid(double stepXY, double sizeXY, double stepZ, double sizeZ)
  {
    TRACE;
    setGrid(-sizeXY, sizeXY, stepXY, -sizeXY, sizeXY, stepXY, -sizeZ, sizeZ, stepZ);
  }

  void GridSearch3D::setGrid(double minX, double maxX, double stepX,
                             double minY, double maxY, double stepY,
                             double minZ, double maxZ, double stepZ)
  {
    TRACE;
    _step[0]=stepX;
    _step[1]=stepY;
    _step[2]=stepZ;
    PrivateVector<double> invStep(3);
    invStep.setValues(1.0/_step[0], 1.0/_step[1], 1.0/_step[2]);
    // Sampling compatible with zero
    _min[0]=qCeil(minX*invStep[0]);
    _min[1]=qCeil(minY*invStep[1]);
    _min[2]=qCeil(minZ*invStep[2]);
    _max[0]=_min[0]+qFloor((maxX-_min[0]*_step[0])*invStep[0]);
    ASSERT(_max[0]-_min[0]>=2);
    _max[1]=_min[1]+qFloor((maxY-_min[1]*_step[1])*invStep[1]);
    ASSERT(_max[1]-_min[1]>=2);
    _max[2]=_min[2]+qFloor((maxZ-_min[2]*_step[2])*invStep[2]);
    ASSERT(_max[2]-_min[2]>=2);
  }

  inline void GridSearch3D::setCurrentPosition(int ix, int iy, int iz)
  {
    _currentPosition[0]=_step[0]*ix;
    _currentPosition[1]=_step[1]*iy;
    _currentPosition[2]=_step[2]*iz;
  }

  void GridSearch3D::globalMax(double absThres)
  {
    TRACE;
    clearMaxima();
#ifdef PROCESS_TIMING
    QElapsedTimer chrono;
    qint64 tgb, trf;
    chrono.start();
#endif
    double val, maxVal=-std::numeric_limits<double>::infinity();
    PrivateVector<int> index(3, 0);
    PrivateVector<int> maxPosition(3, 0);
    int& ix=index[0];
    int& iy=index[1];
    int& iz=index[2];
    for(iy=_min[1]; iy<=_max[1]; iy++) {
      for(ix=_min[0]; ix<=_max[0]; ix++) {
        for(iz=_min[2]; iz<=_max[2]; iz++) {
          val=value(index);
          if(val>maxVal) {
            maxVal=val;
            maxPosition.copyValues(index);
          }
        }
      }
    }
#ifdef PROCESS_TIMING
    tgb=chrono.nsecsElapsed();
#endif
    setCurrentPosition(maxPosition[0], maxPosition[1], maxPosition[2]);
    addMaximum(maxVal);
    FunctionSearch::globalMax(absThres);
#ifdef PROCESS_TIMING
    trf=chrono.nsecsElapsed();
    globalTimeMutex.lock();
    ttgb+=tgb;
    ttrf+=trf-tgb;
    globalTimeMutex.unlock();
#endif
  }

  inline double GridSearch3D::maximumValueZ(Vector<int>& index)
  {
    double val, maxVal=-std::numeric_limits<double>::infinity();
    int& iz=index[2];
    int maxZ=0;
    for(iz=_min[2]; iz<=_max[2]; iz++) {
      val=value(index);
      if(val>maxVal) {
        maxVal=val;
        maxZ=iz;
      }
    }
    iz=maxZ;
    return maxVal;
  }

  /*!
    For each X-Y pair, only one maximum along Z is kept.
    Used to get ellipticity in RTBF method.
  */
  void GridSearch3D::localMax(int nMax, double absThres, double relThres)
  {
    TRACE;
    clearMaxima();
#ifdef PROCESS_TIMING
    QElapsedTimer chrono;
    qint64 tlc, trf;
    chrono.start();
#endif
    double nextLineVal, val;
    int n=_max[0]-_min[0]+1;
    double * previousLine=new double[n]-_min[0];
    double * currentLine=new double[n]-_min[0];
    double * nextLine=new double[n]-_min[0];
    double * swapLine;
    int * currentLineZ=new int[n]-_min[0];
    int * nextLineZ=new int[n]-_min[0];
    int * swapLineZ;
    PrivateVector<int> index(3, 0);
    int& ix=index[0];
    int& iy=index[1];
    iy=_min[1];
    for(ix=_min[0]; ix<=_max[0]; ix++) {
      previousLine[ix]=maximumValueZ(index);
    }
    iy++;
    for(ix=_min[0]; ix<=_max[0]; ix++) {
      currentLine[ix]=maximumValueZ(index);
      currentLineZ[ix]=index[2];
    }
    for(iy++; iy<=_max[1]; iy++) {
      ix=_min[0];
      nextLine[ix]=maximumValueZ(index);
      nextLineZ[ix]=index[2];
      ix++;
      nextLine[ix]=maximumValueZ(index);
      nextLineZ[ix]=index[2];
      for(ix++; ix<=_max[0]; ix++) {
        nextLineVal=maximumValueZ(index);
        nextLine[ix]=nextLineVal;
        nextLineZ[ix]=index[2];
        val=currentLine[ix-1];
        if(val>nextLineVal &&
           val>nextLine[ix-1] &&
           val>nextLine[ix-2] &&
           val>currentLine[ix-2] &&
           val>currentLine[ix] &&
           val>previousLine[ix-2] &&
           val>previousLine[ix-1] &&
           val>previousLine[ix]) {
          // Delayed optimization to avoid replacements (CPU cache) of the pre-computed values
          // required for value computation.
          setCurrentPosition(ix-1, iy-1, currentLineZ[ix-1]);
          addMaximum(val);
        }
      }
      swapLine=previousLine;
      previousLine=currentLine;
      currentLine=nextLine;
      nextLine=swapLine;
      swapLineZ=currentLineZ;
      currentLineZ=nextLineZ;
      nextLineZ=swapLineZ;
    }
    delete [] (previousLine+_min[0]);
    delete [] (currentLine+_min[0]);
    delete [] (nextLine+_min[0]);
    delete [] (currentLineZ+_min[0]);
    delete [] (nextLineZ+_min[0]);
#ifdef PROCESS_TIMING
      tlc=chrono.nsecsElapsed();
#endif
    FunctionSearch::localMax(nMax, absThres, relThres);
#ifdef PROCESS_TIMING
      trf=chrono.nsecsElapsed();
      globalTimeMutex.lock();
      ttlc+=tlc;
      ttrf+=trf-tlc;
      globalTimeMutex.unlock();
#endif
  }

  /*!
    Refines the maximum of the grid in an area defined by current position +/- one step.
    Returns value of the refined maximum
  */
  double GridSearch3D::refineMax(double val)
  {
    TRACE;
#ifdef COMPATIBILITY_3_4_1
    // Gridding is similar but not exactly the same due to arithmetic error.
    // We fix it to ensure a compatible gridding.
    int ix=qRound(_currentPosition[0]/_stepXY)+_halfCountXY;
    int iy=qRound(_currentPosition[1]/_stepXY)+_halfCountXY;
    Point2D min(_minXY+_stepXY*static_cast<double>(ix-1), _minXY+_stepXY*static_cast<double>(iy-1));
    Point2D max(_minXY+_stepXY*static_cast<double>(ix+1), _minXY+_stepXY*static_cast<double>(iy+1));
    Point2D pos=min;
    pos+=max;
    pos*=0.5;
    _currentPosition[0]=pos.x();
    _currentPosition[1]=pos.y();
    Point delta=max;
    delta-=min;
    delta*=0.5;
    delta.setZ(_stepZ);
#endif

    PrivateVector<double> absPrecision(3, 0.0);
    for(int i=0; i<3; i++) {
      if(_relative[i]) {
        absPrecision[i]=fabs(_currentPosition[i]*_precision[i]);
        if(absPrecision[i]==0.0) { // Make sure that precision is not null
          absPrecision[i]=fabs(_precision[i]*(_currentPosition[i]+_step[i]));
        }
      } else {
        absPrecision[i]=_precision[i];
      }
    }

    double x0, x1, x2, y0, y1, y2, z0, z1, z2;
#ifdef COMPATIBILITY_3_4_1
    while(delta.x()>absPrecision[0] ||
          delta.y()>absPrecision[1] ||
          delta.z()>absPrecision[2]) {
      delta*=0.5;
      x0=_currentPosition[0];
      x1=x0-delta.x();
      x2=x0+delta.x();
      y0=_currentPosition[1];
      y1=y0-delta.y();
      y2=y0+delta.y();
      z0=_currentPosition[2];
      z1=z0-delta.z();
      z2=z0+delta.z();
#else
    PrivateVector<double> step(_step);
    while(step.hasElementGreaterThan(absPrecision)) {  // While any of the step components is larger than precision
      step*=0.5;
      // currentPosition may be moved by testForMax, hence it cannot be used
      // after the first testForMax()
      x0=_currentPosition[0];
      x1=x0-step[0];
      x2=x0+step[0];
      y0=_currentPosition[1];
      y1=y0-step[1];
      y2=y0+step[1];
      z0=_currentPosition[2];
      z1=z0-step[2];
      z2=z0+step[2];
#endif
      _testPosition[0]=x1;
      _testPosition[1]=y1;
      _testPosition[2]=z1;
      testForMax(val);
      _testPosition[2]=z0;
      testForMax(val);
      _testPosition[2]=z2;
      testForMax(val);

      _testPosition[1]=y0;
      _testPosition[2]=z1;
      testForMax(val);
      _testPosition[2]=z0;
      testForMax(val);
      _testPosition[2]=z2;
      testForMax(val);

      _testPosition[1]=y2;
      _testPosition[2]=z1;
      testForMax(val);
      _testPosition[2]=z0;
      testForMax(val);
      _testPosition[2]=z2;
      testForMax(val);

      _testPosition[0]=x0;
      _testPosition[1]=y1;
      _testPosition[2]=z1;
      testForMax(val);
      _testPosition[2]=z0;
      testForMax(val);
      _testPosition[2]=z2;
      testForMax(val);

      _testPosition[1]=y0;
      _testPosition[2]=z1;
      testForMax(val);
      // Current position has already been checked, this is val
      _testPosition[2]=z2;
      testForMax(val);

      _testPosition[1]=y2;
      _testPosition[2]=z1;
      testForMax(val);
      _testPosition[2]=z0;
      testForMax(val);
      _testPosition[2]=z2;
      testForMax(val);

      _testPosition[0]=x2;
      _testPosition[1]=y1;
      _testPosition[2]=z1;
      testForMax(val);
      _testPosition[2]=z0;
      testForMax(val);
      _testPosition[2]=z2;
      testForMax(val);

      _testPosition[1]=y0;
      _testPosition[2]=z1;
      testForMax(val);
      _testPosition[2]=z0;
      testForMax(val);
      _testPosition[2]=z2;
      testForMax(val);

      _testPosition[1]=y2;
      _testPosition[2]=z1;
      testForMax(val);
      _testPosition[2]=z0;
      testForMax(val);
      _testPosition[2]=z2;
      testForMax(val);
    }
    return val;
  }

} // namespace QGpCoreMath

