/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-06-10
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "GridSearch2D.h"

namespace QGpCoreMath {

// Get a detailled timing report of the processing
// Run with verbosity level to 3 to get the report.
// Configure with '-D PROCESS_TIMING'
#ifdef PROCESS_TIMING
  static double ttgb=0.0, ttlc=0.0, ttrf=0.0;
  static Mutex globalTimeMutex;
#endif

  /*!
    \class GridSearch2D GridSearch2D.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    GridSearch2D searches for a maximum on X-Y grid.
    The position vector can be larger than a 2D vector.
  */
  GridSearch2D::GridSearch2D(size_t dimensionCount)
    : FunctionSearch(dimensionCount),
      _min(2, 0),
      _max(2, 0),
      _step(2, 0.0)
  {
    TRACE;
    ASSERT(dimensionCount>=2);
  }

  /*!
    Description of destructor still missing
  */
  GridSearch2D::~GridSearch2D()
  {
    TRACE;
  }

  void GridSearch2D::showProcessingTime()
  {
#ifdef PROCESS_TIMING
    App::log(3, tr("GridSearch2D: %1 %2 %3 (ms; global, local, refine)\n")
             .arg(ttgb*1e-6)
             .arg(ttlc*1e-6)
             .arg(ttrf*1e-6));
#endif
  }

  /*!
    Create a symetric gridding identical along X and Y.
  */
  void GridSearch2D::setGrid(double step, double size)
  {
    TRACE;
    setGrid(-size, size, step, -size, size, step);
  }

  void GridSearch2D::setGrid(double minX, double maxX, double stepX,
                             double minY, double maxY, double stepY)
  {
    TRACE;
    _step[0]=stepX;
    _step[1]=stepY;
    PrivateVector<double> invStep(2);
    invStep.setValues(1.0/_step[0], 1.0/_step[1]);
    // Sampling compatible with zero
    _min[0]=qCeil(minX*invStep[0]);
    _min[1]=qCeil(minY*invStep[1]);
    _max[0]=_min[0]+qFloor((maxX-_min[0]*_step[0])*invStep[0]);
    ASSERT(_max[0]-_min[0]>=2);
    _max[1]=_min[1]+qFloor((maxY-_min[1]*_step[1])*invStep[1]);
    ASSERT(_max[1]-_min[1]>=2);
  }

  inline void GridSearch2D::setCurrentPosition(int ix, int iy)
  {
    _currentPosition[0]=_step[0]*ix;
    _currentPosition[1]=_step[1]*iy;
  }

  void GridSearch2D::globalMax(double absThres)
  {
    TRACE;
    clearMaxima();
#ifdef PROCESS_TIMING
    QElapsedTimer chrono;
    qint64 tgb, trf;
    chrono.start();
#endif
    double val, maxVal=-std::numeric_limits<double>::infinity();
    PrivateVector<int> index(2, 0);
    PrivateVector<int> maxPosition(2, 0);
    int& ix=index[0];
    int& iy=index[1];
    for(iy=_min[1]; iy<=_max[1]; iy++) {
      for(ix=_min[0]; ix<=_max[0]; ix++) {
        val=value(index);
        if(val>maxVal) {
          maxVal=val;
          maxPosition.copyValues(index);
        }
      }
    }
#ifdef PROCESS_TIMING
    tgb=chrono.nsecsElapsed();
#endif
    setCurrentPosition(maxPosition[0], maxPosition[1]);
    addMaximum(maxVal);
    FunctionSearch::globalMax(absThres);
#ifdef PROCESS_TIMING
    trf=chrono.nsecsElapsed();
    globalTimeMutex.lock();
    ttgb+=tgb;
    ttrf+=trf-tgb;
    globalTimeMutex.unlock();
#endif
  }

  void GridSearch2D::localMax(int nMax, double absThres, double relThres)
  {
    TRACE;
    clearMaxima();
#ifdef PROCESS_TIMING
    QElapsedTimer chrono;
    qint64 tlc, trf;
    chrono.start();
#endif
    double nextLineVal, val;
    int n=_max[0]-_min[0]+1;
    double * previousLine=new double[n]-_min[0];
    double * currentLine=new double[n]-_min[0];
    double * nextLine=new double[n]-_min[0];
    double * swapLine;
    PrivateVector<int> index(2, 0);
    int& ix=index[0];
    int& iy=index[1];
    iy=_min[1];
    for(ix=_min[0]; ix<=_max[0]; ix++) {
      previousLine[ix]=value(index);
    }
    iy++;
    for(ix=_min[0]; ix<=_max[0]; ix++) {
      currentLine[ix]=value(index);
    }
    for(iy++; iy<=_max[1]; iy++) {
      ix=_min[0];
      nextLine[ix]=value(index);
      ix++;
      nextLine[ix]=value(index);
      for(ix++; ix<=_max[0]; ix++) {
        nextLineVal=value(index);
        nextLine[ix]=nextLineVal;
        val=currentLine[ix-1];
        if(val>nextLineVal &&
           val>nextLine[ix-1] &&
           val>nextLine[ix-2] &&
           val>currentLine[ix-2] &&
           val>currentLine[ix] &&
           val>previousLine[ix-2] &&
           val>previousLine[ix-1] &&
           val>previousLine[ix]) {
          // Delayed optimization to avoid replacements (CPU cache) of the pre-computed values
          // required for value computation.
          setCurrentPosition(ix-1, iy-1);
          addMaximum(val);
        }
      }
      swapLine=previousLine;
      previousLine=currentLine;
      currentLine=nextLine;
      nextLine=swapLine;
    }
    delete [] (previousLine+_min[0]);
    delete [] (currentLine+_min[0]);
    delete [] (nextLine+_min[0]);
#ifdef PROCESS_TIMING
      tlc=chrono.nsecsElapsed();
#endif
    FunctionSearch::localMax(nMax, absThres, relThres);
#ifdef PROCESS_TIMING
      trf=chrono.nsecsElapsed();
      globalTimeMutex.lock();
      ttlc+=tlc;
      ttrf+=trf-tlc;
      globalTimeMutex.unlock();
#endif
  }

  /*!
    Refines the maximum of the grid in an area defined by current position +/- one step.
    Returns value of the refined maximum
  */
  double GridSearch2D::refineMax(double val)
  {
    TRACE;
#ifdef COMPATIBILITY_3_4_1
    // Gridding is similar but not exactly the same due to arithmetic error.
    // We fix it to ensure a compatible gridding.
    int ix=qRound(_currentPosition[0]/_step[0])+_halfCount;
    int iy=qRound(_currentPosition[1]/_step[1])+_halfCount;
    Point2D min(_min[0]+_step[0]*static_cast<double>(ix-1), _min[1]+_step[1]*static_cast<double>(iy-1));
    Point2D max(_min[0]+_step[0]*static_cast<double>(ix+1), _min[1]+_step[1]*static_cast<double>(iy+1));
    Point2D pos=min;
    pos+=max;
    pos*=0.5;
    _currentPosition[0]=pos.x();
    _currentPosition[1]=pos.y();
    Point delta=max;
    delta-=min;
    delta*=0.5;
#endif

    PrivateVector<double> absPrecision(2, 0.0);
    for(int i=0; i<2; i++) {
      if(_relative[i]) {
        absPrecision[i]=fabs(_currentPosition[i]*_precision[i]);
        if(absPrecision[i]==0.0) { // Make sure that precision is not null
          absPrecision[i]=fabs(_precision[i]*(_currentPosition[i]+_step[i]));
        }
      } else {
        absPrecision[i]=_precision[i];
      }
    }

#ifdef COMPATIBILITY_3_4_1
    while(delta.x()>absPrecisionX ||
          delta.y()>absPrecisionY) {
      delta.setX(0.5*delta.x());
      delta.setY(0.5*delta.y());
      x0=_currentPosition[0];
      x1=x0-delta.x();
      x2=x0+delta.x();
      y0=_currentPosition[1];
      y1=y0-delta.y();
      y2=y0+delta.y();
#else
   PrivateVector<double> step(_step);
   double x0, y0, x1, x2, y1, y2;
   double val0=val;
   while(step.hasElementGreaterThan(absPrecision)) {  // While any of the step components is larger than precision
      step*=0.5;
      // currentPosition may be moved by testForMax, hence it cannot be used
      // after the first testForMax()
      x0=_currentPosition[0];
      x1=x0-step[0];
      x2=x0+step[0];
      y0=_currentPosition[1];
      y1=y0-step[0];
      y2=y0+step[0];
#endif
      _testPosition[0]=x1;
      _testPosition[1]=y1;
      testForMax(val);
      _testPosition[1]=y0;
      testForMax(val);
      _testPosition[1]=y2;
      testForMax(val);

      _testPosition[0]=x0;
      _testPosition[1]=y1;
      testForMax(val);
      // Current position has already been checked, this is val
      _testPosition[1]=y2;
      testForMax(val);

      _testPosition[0]=x2;
      _testPosition[1]=y1;
      testForMax(val);
      _testPosition[1]=y0;
      testForMax(val);
      _testPosition[1]=y2;
      testForMax(val);
    }
    // All refine steps did not change the best value
    // If the dimension of x vector is larger than 2,
    // the other dimensions must be updated.
    if(val==val0 && _testPosition.count()>2) {
      value(_currentPosition);
    }
    return val;
  }

} // namespace QGpCoreMath

