/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-06-15
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "GridSearch1D.h"

namespace QGpCoreMath {

// Get a detailled timing report of the processing
// Run with verbosity level to 3 to get the report.
// Configure with '-D PROCESS_TIMING'
#ifdef PROCESS_TIMING
  static double ttgb=0.0, ttlc=0.0, ttrf=0.0;
  static Mutex globalTimeMutex;
#endif

  /*!
    \class GridSearch1D GridSearch1D.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  GridSearch1D::GridSearch1D(size_t dimensionCount)
    : FunctionSearch(dimensionCount)
  {
    TRACE;
    _step=0.0;
    _min=0;
    _max=0;
  }

  /*!
    Description of destructor still missing
  */
  GridSearch1D::~GridSearch1D()
  {
    TRACE;
  }

  void GridSearch1D::showProcessingTime()
  {
#ifdef PROCESS_TIMING
    App::log(3, tr("GridSearch1D: %1 %2 %3 (ms; global, local, refine)\n")
             .arg(ttgb*1e-6)
             .arg(ttlc*1e-6)
             .arg(ttrf*1e-6));
#endif
  }

  void GridSearch1D::setGrid(double min, double max, double step)
  {
    TRACE;
    _step=step;
    double invStep=1.0/_step;
    _min=qCeil(min*invStep);
    _max=_min+qFloor((max-_min*_step)*invStep);
    ASSERT(_max-_min>=2);  // At least 3 positions
  }

  inline void GridSearch1D::setCurrentPosition(int i)
  {
    _currentPosition[0]=_step*i;
  }

  /*!
    Search all angles for the maximum and refine it
  */
  void GridSearch1D::globalMax(double absThres)
  {
    TRACE;
    clearMaxima();
#ifdef PROCESS_TIMING
    QElapsedTimer chrono;
    qint64 tgb, trf;
    chrono.start();
#endif
    double val, maxVal=-std::numeric_limits<double>::infinity();
    int maxPosition=0;
    for(int ix=_min; ix<=_max; ix++) {
      setCurrentPosition(ix);
      val=value(_currentPosition);
      if(val>maxVal) {
        maxVal=val;
        maxPosition=ix;
      }
    }
#ifdef PROCESS_TIMING
     tgb=chrono.nsecsElapsed();
#endif
    setCurrentPosition(maxPosition);
    addMaximum(maxVal);
    FunctionSearch::globalMax(absThres);
#ifdef PROCESS_TIMING
    trf=chrono.nsecsElapsed();
    globalTimeMutex.lock();
    ttgb+=tgb;
    ttrf+=trf-tgb;
    globalTimeMutex.unlock();
#endif
  }

  /*!
    Description still missing
  */
  void GridSearch1D::localMax(int nMax, double absThres, double relThres)
  {
    TRACE;
    clearMaxima();
#ifdef PROCESS_TIMING
    QElapsedTimer chrono;
    qint64 tlc, trf;
    chrono.start();
#endif
    double previous, current, next;
    int ix=_min;
    setCurrentPosition(ix);
    previous=value(_currentPosition);
    ix++;
    setCurrentPosition(ix);
    current=value(_currentPosition);
    for(ix++; ix<=_max; ix++) {
      setCurrentPosition(ix);
      next=value(_currentPosition);
      if(current>next &&
        current>previous) {
        setCurrentPosition(ix-1);
        addMaximum(current);
      }
      previous=current;
      current=next;
    }
#ifdef PROCESS_TIMING
    tlc=chrono.nsecsElapsed();
#endif
    FunctionSearch::localMax(nMax, absThres, relThres);
#ifdef PROCESS_TIMING
    trf=chrono.nsecsElapsed();
    globalTimeMutex.lock();
    ttlc+=tlc;
    ttrf+=trf-tlc;
    globalTimeMutex.unlock();
#endif
  }

  /*!
    Refines the maximum of the grid in an area defined by current position +/- one step.
    Returns value of the refined maximum
  */
  double GridSearch1D::refineMax(double val)
  {
    TRACE;
    double absPrecision;
    if(_relative[0]) {
      absPrecision=fabs(_currentPosition[0]*_precision[0]);
      if(absPrecision==0.0) { // Make sure that precision is not null
        absPrecision=fabs(_precision[0]*(_currentPosition[0]+_step));
      }
    } else {
      absPrecision=_precision[0];
    }

    double x0;
    double step=_step;
    while(step>absPrecision) {
      step*=0.5;
      x0=_currentPosition[0];
      _testPosition[0]=x0-step;
      testForMax(val);
      _testPosition[0]=x0+step;
      testForMax(val);
    }
    return val;
  }

} // namespace QGpCoreMath

