/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-11-12
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef FUNCTIONSEARCH_H
#define FUNCTIONSEARCH_H

#include "QGpCoreMathDLLExport.h"
#include "AbstractFunction.h"
#include "PrivateVector.h"

namespace QGpCoreMath {

  class QGPCOREMATH_EXPORT FunctionSearch
  {
  public:
    FunctionSearch(int dimensionCount);
    virtual ~FunctionSearch();

    void setFunction(AbstractFunction * f);
    AbstractFunction * function() const {return _function;}
    AbstractFunction * takeFunction();

    double value(const Vector<int>& index) const {return _function->value(index);}
    double value(Vector<double>& x) const {return _function->value(x);}

    const Vector<double>& currentPosition() const {return _currentPosition;}
    virtual void globalMax(double absThres=0.0);
    virtual void localMax(int nMax, double absThres=0.0, double relThres=0.0);
    void unique();

    int maximaCount() const {return _maxima.count();}
    FunctionMaximaIterator begin() {return _maxima.begin();}
    FunctionMaximaIterator end() {return _maxima.end();}

    void setRelativePrecision(int axis, double p) {_precision[axis]=p; _relative[axis]=true;}
    void setAbsolutePrecision(int axis, double p) {_precision[axis]=p; _relative[axis]=false;}
    double precision(int axis) const {return _precision[axis];}
    bool relative(int axis) const {return _relative[axis];}

    void debugScan(const Vector<double>& min, const Vector<double>& max, int n);
  protected:
    void clearMaxima();
    void addMaximum(double val);
    virtual double refineMax(double val);
    inline void testForMax(double& maxVal);
    static inline double lowLimit(int i, double * axis);
    static inline double highLimit(int i, double * axis, int nAxis);

    Vector<double> _currentPosition;
    PrivateVector<double> _testPosition;
    AbstractFunction * _function;
    PrivateVector<bool> _relative;
    PrivateVector<double> _precision;
    VectorList<FunctionSearchMaximum *> _maxima;
  };

  inline double FunctionSearch::lowLimit(int i, double * axis)
  {
    return i>0 ? axis[i-1] : axis[0];
  }

  inline double FunctionSearch::highLimit(int i, double * axis, int nAxis)
  {
    return i<nAxis-1 ? axis[i+1] : axis[nAxis-1];
  }

  inline void FunctionSearch::testForMax(double& maxVal)
  {
    TRACE;
    double val=value(_testPosition);
    if(val>maxVal) {
      maxVal=val;
      _currentPosition.copyValues(_testPosition);
    }
  }

} // namespace QGpCoreMath

#endif // FUNCTION2SEARCH.H
