/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  QGpCoreMath is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  QGpCoreMath is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-01-13
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "DelaunayTriangulation.h"
#include "Line2D.h"

namespace QGpCoreMath {

  void DelaunayTriangle::super(const QList<Point2D>& points)
  {
    Triangle::super(points);
    setCircumcircle();
    setSurface();
  }

  /*!
    \class DelaunayTriangulation DelaunayTriangulation.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  DelaunayTriangulation::DelaunayTriangulation()
  {
    TRACE;
  }

  /*!
    Description of destructor still missing
  */
  DelaunayTriangulation::~DelaunayTriangulation()
  {
    TRACE;
  }

  /*!
    Implementation of Bowyer-Watson algorithm
    https://en.wikipedia.org/wiki/Bowyer%E2%80%93Watson_algorithm

    Robustness modifications to support imprecise circumcircle computation.
    For a robust circumcircle see http://www.cs.cmu.edu/~quake/robust.html
  */
  void DelaunayTriangulation::init(const QList<Point2D>& points)
  {
    DelaunayTriangle superTriangle;
    // A super triangle that contains all points
    superTriangle.super(points);
    double superSurface=superTriangle.surface();
    _triangles.append(superTriangle);
    int np=points.count();
    for(int ip=0; ip<np; ip++) {
      const Point2D& newPoint=points.at(ip);
      int n=_triangles.count();

      // Select all triangle whose circumcircle approximately include the new point.
      VectorList<int> badTriangleCandidates;
      for(int i=0; i<n; i++) {
        if(_triangles.at(i).circumcircle().squareContains(newPoint, 1.0+1e-9)) {
          badTriangleCandidates.append(i);
        }
      }

      // Select those whose circumcircle strictly include the new point.
      n=badTriangleCandidates.count();
      VectorList<int> badTriangles;
      VectorList<int> criticalTriangles;
      for(int i=0; i<n; i++) {
        if(_triangles.at(badTriangleCandidates.at(i)).circumcircle().squareContains(newPoint, 1.0)) {
          badTriangles.append(badTriangleCandidates.at(i));
        } else {
          criticalTriangles.append(badTriangleCandidates.at(i));
        }
      }

      // Recover approximate triangles that share two edges with the strict list
      if(!criticalTriangles.isEmpty()) {

        // If the triangles whose circumcircle approximately include the new points
        // share at least two edges with the triangles whose circumcircle strictly
        // include the new points, they must be kept. The approximate inclusion is
        // due to precision errors.
        n=criticalTriangles.count();
        for(int i=0; i<n; i++) {
          int currentIndex=criticalTriangles.at(i);
          const Triangle& t=_triangles.at(currentIndex);
          int commonEdges=0;
          for(int j=badTriangleCandidates.count()-1; j>=0; j--) {
            int neighborIndex=badTriangleCandidates.at(j);
            if(neighborIndex!=currentIndex && t.isNeighbor(_triangles.at(neighborIndex))) {
               commonEdges++;
               if(commonEdges==2) {
                 badTriangles.append(criticalTriangles.at(i));
                 break;
               }
            }
          }
        }
        std::sort(badTriangles.begin(), badTriangles.end());
        if(App::verbosity()>=1) {
          App::log(tr("Critical bad triangle selection when inserting point %1 (%2)\n")
                   .arg(ip)
                   .arg(newPoint.toString()));
          int strictBadTriangles=badTriangles.count();
          if(badTriangles.count()>strictBadTriangles || App::verbosity()>=3) {
            App::log(tr("  Approximately included by %1 triangles\n"
                        "  Strictly included by %2 triangles\n"
                        "     ---> %1 addtional triangles\n")
                     .arg(criticalTriangles.count())
                     .arg(strictBadTriangles)
                     .arg(badTriangles.count()-strictBadTriangles));
          } else {
            App::log(tr("  No additional triangle\n"));
          }
          if(App::verbosity()>=4) {
            QString s;
            for(int i=badTriangles.count()-1; i>=0; i--) {
              s+="   ";
              s+=_triangles.at(badTriangles.at(i)).toString();
              s+="\n";
            }
            App::log(tr("List of bad triangles (strict):\n%1").arg(s));
            s.clear();
            for(int i=criticalTriangles.count()-1; i>=0; i--) {
              s+="   ";
              s+=_triangles.at(criticalTriangles.at(i)).toString();
              s+="\n";
            }
            App::log(tr("List of critical triangles:\n%1").arg(s));
          }
        }
      }

      VectorList<Edge> polygon;
      Edge edge;
      for(int i=badTriangles.count()-1; i>=0; i--) {
        const DelaunayTriangle& t=_triangles.at(badTriangles.at(i));
        edge.p1=t.vertex(0);
        edge.p2=t.vertex(1);
        if(indexOf(edge, badTriangles, i)<0) {
          polygon.append(edge);
        }
        edge.p2=t.vertex(2);
        if(indexOf(edge, badTriangles, i)<0) {
          polygon.append(edge);
        }
        edge.p1=t.vertex(1);
        if(indexOf(edge, badTriangles, i)<0) {
          polygon.append(edge);
        }
      }
      QString badTriangleUserList;
      if(App::verbosity()>=2) {
        for(int i=badTriangles.count()-1; i>=0; i--) {
          badTriangleUserList+="   ";
          badTriangleUserList+=_triangles.at(badTriangles.at(i)).toString();
          badTriangleUserList+="\n";
        }
      }

      // badTriangle is naturally sorted, start to remove last ones first
      for(int i=badTriangles.count()-1; i>=0; i--) {
        _triangles.removeAt(badTriangles.at(i));
      }
      for(int i=polygon.count()-1; i>=0; i--) {
        const Edge& edge=polygon.at(i);
        DelaunayTriangle nt(newPoint, edge.p1, edge.p2);
        if(nt.circumcircle().radius()>0.0) {
          _triangles.append(nt);
        } else {
          App::log(tr("Degenerated triangle (null surface) while adding point %1\n").arg(ip));
        }
      }

      if(fabs(surface()-superSurface)>superSurface*1e-10) {
        App::log(tr("Overlapping triangles after inserting point %1 (%2)\n")
                 .arg(ip)
                 .arg(newPoint.toString()));
        APP_LOG(2, tr("List of bad triangles:\n%1").arg(badTriangleUserList));
        return;
      }
    }
    // Remove all triangles containing vertices from super triangle
    for(int i=_triangles.count()-1; i>=0; i--) {
      const DelaunayTriangle& t=_triangles.at(i);
      if(t.hasVertex(superTriangle.vertex(0)) ||
         t.hasVertex(superTriangle.vertex(1)) ||
         t.hasVertex(superTriangle.vertex(2))) {
        _triangles.removeAt(i);
      }
    }
  }

  int DelaunayTriangulation::indexOf(const Edge& edge, const VectorList<int>& triangles, int current) const
  {
    int i;
    for(i=triangles.count()-1; i>current; i--) {
      if(_triangles.at(triangles.at(i)).hasEdge(edge.p1, edge.p2)) {
        return i;
      }
    }
    for(i=current-1; i>=0; i--) {
      if(_triangles.at(triangles.at(i)).hasEdge(edge.p1, edge.p2)) {
        return i;
      }
    }
    return -1;
  }

  /*!
    List of triangles attached to point \a p.
  */
  VectorList<DelaunayTriangle> DelaunayTriangulation::triangles(const Point2D& p) const
  {
    VectorList<DelaunayTriangle> triangles;
    for(int i=_triangles.count()-1; i>=0; i--) {
      const DelaunayTriangle& t=_triangles.at(i);
      if(t.hasVertex(p)) {
        triangles.append(t);
      }
    }
    return triangles;
  }

  VectorList<Curve<Point2D>> DelaunayTriangulation::triangleEdges() const
  {
    VectorList<Curve<Point2D>> segments;
    Curve<Point2D> segment(2);
    int n=_triangles.count();
    for(int i=0; i<n; i++) {
      const DelaunayTriangle& t=_triangles.at(i);
      segment.at(0)=t.vertex(0);
      segment.at(1)=t.vertex(1);
      segments.append(segment);
      segment.at(1)=t.vertex(2);
      segments.append(segment);
      segment.at(0)=t.vertex(1);
      segments.append(segment);
    }
    return segments;
  }

  VectorList<Curve<Point2D>> DelaunayTriangulation::voronoiCells(const Rect& limits) const
  {
    VectorList<Curve<Point2D>> segments;
    Curve<Point2D> segment(2);
    int n=_triangles.count();
    VectorList<int> triangleIndexes(n);
    // Indexes are used by indexOf(), but in this case indirect and direct indexes are the same.
    for(int i=0; i<n; i++) {
      triangleIndexes[i]=i;
    }
    Edge edge;
    for(int i=0; i<n; i++) {
      const DelaunayTriangle& t=_triangles.at(i);
      const Point2D& center=t.circumcircle().center();
      segment.at(0)=center;
      edge.p1=t.vertex(0);
      edge.p2=t.vertex(1);
      segment.at(1)=segmentEnd(edge, t.vertex(2), center, i, triangleIndexes, limits);
      if(segment.at(1)!=center) {
        segments.append(segment);
      }
      edge.p2=t.vertex(2);
      segment.at(1)=segmentEnd(edge, t.vertex(1), center, i, triangleIndexes, limits);
      if(segment.at(1)!=center) {
        segments.append(segment);
      }
      edge.p1=t.vertex(1);
      segment.at(1)=segmentEnd(edge, t.vertex(0), center, i, triangleIndexes, limits);
      if(segment.at(1)!=center) {
        segments.append(segment);
      }
    }
    // Sort all individual segments to ensure that they are all organized the same way
    for(int i=segments.count()-1; i>=0; i--) {
      Curve<Point2D>& s=segments[i];
      std::sort(s.begin(), s.end());
    }
    // Sort and remove double entries
    std::sort(segments.begin(), segments.end());
    unique(segments);
    return segments;
  }

  /*!
    Returns segment end that cross \a edge inside a triangle composed of \a edge and \a vertex.
    \a center is the center of the circumcircle. \a index is the index of the triangle.
    A segment may connect two triangles or be external. External segments are limited inside \a limits.
    The related triangle is searched in \a triangleIndexes excluding \a index which the is the
    current triangle of interest.
  */
  Point2D DelaunayTriangulation::segmentEnd(const Edge& edge, const Point& vertex, const Point2D& center,
                                            int index, const VectorList<int>& triangleIndexes,
                                            const Rect& limits) const
  {
    int j=indexOf(edge, triangleIndexes, index);
    if(j>=0) {
      return _triangles.at(j).circumcircle().center();
    } else if(limits.includes(center)) {
      Line2D line(edge.p1, edge.p2);
      Line2D perp(line);
      perp.perpendicular(center);
      Point2D pl1, pl2;
      limits.intersect(perp, pl1, pl2);
      // We have to choose one of the two points.
      // The good one is oppposite to vertex.
      if(line.isOnSameSide(pl1, vertex)) {
        return pl2;
      } else {
        return pl1;
      }
    } else {
      return center; // a null length segment
    }
  }

  double DelaunayTriangulation::surface() const
  {
    double sum=0.0;
    for(int i=_triangles.count()-1; i>=0; i--) {
      sum+=_triangles.at(i).surface();
    }
    return sum;
  }

} // namespace QGpCoreMath

