/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-06-28
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef CURVESPLITTER_H
#define CURVESPLITTER_H

#include <QGpCoreTools.h>

#include "QGpCoreMathDLLExport.h"
#include "TriangularMatrix.h"
#include "Cluster.h"
#include "Point2D.h"

namespace QGpCoreMath {

  template <class pointType> class Curve;

  class QGPCOREMATH_EXPORT CurveSplitter
  {
    TRANSLATIONS("CurveSplitter")
  public:
    CurveSplitter();
    ~CurveSplitter();

    void addPair(int i1, int i2, double x0, double y0, double dx, double y1);
    void clusterize(double maxErr);
    void validSegments(double maxErr, const Curve<Point2D>& curve);
    void continuousSegments(const VectorList<double>& x);
  private:
    class Item;

    int pointCount() const;
    TriangularMatrix proximity(int pointCount) const;
    int indexOf(int index) const;
    Item * item(int index) {return &_items[index];}
    void validSegments(Cluster::Node * clusters, double maxErr, const Curve<Point2D>& curve);
    void continuousSegments(Cluster::Node * clusters,
                            const VectorList<double>& x,
                            const QMap<double, int>& xMap);

    class ItemFriend
    {
    public:
      ItemFriend() {
        _index=0;
        _slope=0.0;
        _dx=0.0;
        _y=0.0;
      }
      ItemFriend(int i, double slope, double dx, double value) {
        _index=i;
        _slope=slope;
        _dx=dx;
        _y=value;
      }

      bool operator<(const ItemFriend& o) const {return _index<o._index;}
      bool operator==(const ItemFriend& o) const {return _index==o._index;}

      int _index;
      double _slope, _dx, _y;
    };

    class FriendMap
    {
    public:
      FriendMap(int n);
      ~FriendMap();

      void add(int index) {_map[index]=true; _list.append(index);}

      int count() const {return _list.count();}
      bool operator==(const FriendMap& o) const;
      QString toString() const;
      double distanceTo(const FriendMap& o) const;
    private:
      bool * _map;
      VectorList<int> _list;
    };

    class Item
    {
      TRANSLATIONS("CurveSplitter::Item")
    public:
      Item();
      ~Item();

      void sortFriends(double maxErr, int pointCount);
      double distanceTo(const Item& o) const;
      static bool lessThan(const QPair<int, double>& f1, const QPair<int, double>& f2);

      int _index;
      double _x, _y;
      VectorList<ItemFriend> _friends;
      QList<FriendMap *> _goodFriends;
    };

    VectorList<Item> _items;
    Cluster::Node * _clusters;
    VectorList<VectorList<int>> _validClusters;

  public:
    int clusterCount() const {return _validClusters.count();}
    VectorList<int> pointIndex(int cluster) const {return _validClusters.at(cluster);}
  };

} // namespace QGpCoreMath

#endif // CURVESPLITTER_H

