/***************************************************************************
**
**  This file is part of QGpCoreMath.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2017-06-28
**  Copyright: 2017-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <QGpCoreTools.h>

#include "CurveSplitter.h"
#include "Curve.h"

namespace QGpCoreMath {

  /*!
    \class CurveSplitter CurveSplitter.h
    \brief Brief description of class still missing

    Store relations between indexes (for instance of a VectorList). A value is assigned
    to all added pairs of indexes. If some values are similar, indexes are considered
    as friends. Clusters are built with reciprocical friends.

    This object was originaly designed to clusterize a Curve made of randomly sorted
    samples distributed on several sub-curves (\sa Curve::split()).
  */

  CurveSplitter::Item::Item()
  {
    TRACE;
    _index=-1;
  }

  CurveSplitter::Item::~Item()
  {
    TRACE;
    qDeleteAll(_goodFriends);
  }

  /*!
    Each item has friends. Friends are in the X neighborhood.
    Each friend has a y value and a slope calculated from the point of this item.
    This function sorts friends by considering all pairs of friends. If, using the slope
    of the first friend, the y value of the second can be predicted within \a maxErr,
    they are both added to the collection. Each point can belong to several friend networks.
  */
  void CurveSplitter::Item::sortFriends(double maxErr, int pointCount)
  {
    TRACE;
    ASSERT(_goodFriends.isEmpty());
    if(_friends.isEmpty()) {
      return;
    }
    int n=_friends.count();
    for(int i=0; i<n; i++) {
      FriendMap * collectedFriends=new FriendMap(pointCount);
      collectedFriends->add(_index); // I'm my best friend
      const ItemFriend& f1=_friends.at(i);
      collectedFriends->add(f1._index);
      // All combinations are tested... justify why there is no reprocity (j from i+1)
      for(int j=0; j<i; j++) {
        const ItemFriend& f2=_friends.at(j);
        if(fabs(1.0-(_y+f1._slope*f2._dx)/f2._y)<maxErr) {
          collectedFriends->add(f2._index);
        }
      }
      for(int j=i+1; j<n; j++) {
        const ItemFriend& f2=_friends.at(j);
        if(fabs(1.0-(_y+f1._slope*f2._dx)/f2._y)<maxErr) {
          collectedFriends->add(f2._index);
        }
      }
      if(_goodFriends.isEmpty()) {
        _goodFriends.append(collectedFriends);
      } else {
        int n=_goodFriends.first()->count();
        if(collectedFriends->count()>n) {
          qDeleteAll(_goodFriends);
          _goodFriends.clear();
          _goodFriends.append(collectedFriends);
        } else if(collectedFriends->count()==n) {
          int i;
          for(i=_goodFriends.count()-1; i>=0; i--) {
            if(*collectedFriends==*_goodFriends.at(i)) {
              break;
            }
          }
          if(i<0) {
            _goodFriends.append(collectedFriends);
          } else {
            delete collectedFriends;
          }
        } else {
          delete collectedFriends;
        }
      }
    }
    if(App::verbosity()>=2) {
      App::log(tr("Good friends for item index %1\n").arg(_index));
      for(int i=0; i<_goodFriends.count(); i++) {
        App::log(tr("  list %1\n").arg(i));
        App::log(_goodFriends.at(i)->toString());
      }
    }
  }

  double CurveSplitter::Item::distanceTo(const Item& o) const
  {
    double maxScore=0.0;
    for(int i=_goodFriends.count()-1; i>=0; i--) {
      const FriendMap * f=_goodFriends.at(i);
      for(int j=o._goodFriends.count()-1; j>=0; j--) {
        double s=f->distanceTo(*o._goodFriends.at(j));
        if(s>maxScore) {
          maxScore=s;
        }
      }
    }
    return maxScore;
  }

  CurveSplitter::FriendMap::FriendMap(int n)
  {
    _map=new bool[n];
    memset(_map, false, n*sizeof(bool));
  }

  CurveSplitter::FriendMap::~FriendMap()
  {
    delete [] _map;
  }

  QString CurveSplitter::FriendMap::toString() const
  {
    QString s;
    int n=_list.count();
    for(int i=0; i<n; i++) {
      s+=tr("    index %1\n").arg(_list.at(i));
    }
    return s;
  }

  bool CurveSplitter::FriendMap::operator==(const FriendMap& o) const
  {
    for(int i=_list.count()-1; i>=0; i--) {
      if(!o._map[_list.at(i)]) {
        return false;
      }
    }
    for(int i=o._list.count()-1; i>=0; i--) {
      if(!_map[_list.at(i)]) {
        return false;
      }
    }
    return true;
  }

  /*!
    The number of common friends is counted and normalized by the size of the smallest
    friend list. The inverse is returned.
    The distance is the same if 'this' is replaced by \a o.
  */
  double CurveSplitter::FriendMap::distanceTo(const FriendMap& o) const
  {
    int n=0;
    if(_list.count()<o._list.count()) {
      for(int i=_list.count()-1; i>=0; i--) {
        if(o._map[_list.at(i)]) {
          n++;
        }
      }
      return static_cast<double>(_list.count())/static_cast<double>(n);
    } else {
      for(int i=o._list.count()-1; i>=0; i--) {
        if(_map[o._list.at(i)]) {
          n++;
        }
      }
      return static_cast<double>(o._list.count())/static_cast<double>(n);
    }
  }

  /*!
    Description of constructor still missing
  */
  CurveSplitter::CurveSplitter()
  {
    TRACE;
    _clusters=nullptr;
  }

  /*!
    Description of destructor still missing
  */
  CurveSplitter::~CurveSplitter()
  {
    TRACE;
    delete _clusters;
  }

  int CurveSplitter::indexOf(int index) const
  {
    for(int i=_items.count()-1; i>=0; i--) {
      if(index==_items.at(i)._index) {
        return i;
      }
    }
    return -1;
  }

  /*!

  */
  void CurveSplitter::addPair(int i1, int i2, double x0, double y0, double dx, double y1)
  {
    int i=indexOf(i1);
    if(i<0) {
      i=_items.count();
      _items.resize(i+1);
      _items[i]._index=i1;
      _items[i]._x=x0;
      _items[i]._y=y0;
      App::log(2, tr("Add new item at %1 with value %2\n").arg(i1).arg(y0));
    }
    double slope=(y1-y0)/dx;
    _items[i]._friends.append(ItemFriend(i2, slope, dx, y1));
    App::log(2, tr("Add pair between %1 and %2 with slope %3 dx %4 value %5\n")
                        .arg(i1).arg(i2).arg(slope).arg(dx).arg(y1));
  }

  int CurveSplitter::pointCount() const
  {
    int n=_items.count();
    // Get the maximum point index
    int np=n;
    for(int i=0; i<n; i++) {
      int ip=_items.at(i)._index;
      if(ip>=np) {
        np=ip+1;
      }
    }
    return np;
  }

  /*!
    Compute distances of all couples.
    The matrix dimension may be larger than the number of items.
    Indexes to matrix elements are the point index.
  */
  TriangularMatrix CurveSplitter::proximity(int pointCount) const
  {
    int n=_items.count();
    // Build matrix
    TriangularMatrix m(pointCount);
    for(int i=0; i<n; i++) {
      int ip=_items.at(i)._index;
      for(int j=i+1; j<n; j++) {
        int jp=_items.at(j)._index;
        m.at(ip, jp)=_items.at(i).distanceTo(_items.at(j));
      }
    }
    if(App::verbosity()>=4) {
      QTextStream(stdout) << m.toGridString() << Qt::endl;
    }
    return m;
  }

  /*!
    \a maxErr is a relative error.
  */
  void CurveSplitter::clusterize(double maxErr)
  {
    TRACE;
    int np=pointCount();
    for(int i=_items.count()-1; i>=0; i--) {
      _items[i].sortFriends(maxErr, np);
    }
    if(np>0) {
      Cluster c(proximity(np));
      delete _clusters;
      _clusters=c.singleLinkage();
      if(App::verbosity()>=3) {
        XMLHeader hdr(_clusters);
        QTextStream(stdout) << hdr.xml_saveString() << Qt::endl;
      }
    }
  }

  /*!
    \a x is the unsorted list of x values
  */
  void CurveSplitter::validSegments(double maxErr, const Curve<Point2D>& curve)
  {
    _validClusters.clear();
    if(_clusters) {
      validSegments(_clusters, maxErr, curve);
    }
  }

  void CurveSplitter::validSegments(Cluster::Node * clusters, double maxErr, const Curve<Point2D>& curve)
  {
    VectorList<int> indexList=clusters->indexList();
    int n=indexList.count();
    if(n>1) {  // Get rid of very small clusters
      std::sort(indexList.begin(), indexList.end());
      for(int i=2; i<n; i++) {
        const Point2D& p0=curve.constAt(indexList.at(i-2));
        const Point2D& p1=curve.constAt(indexList.at(i-1));
        const Point2D& p2=curve.constAt(indexList.at(i));
        if(fabs(1.0-(p0.y()+(p1.y()-p0.y())/(p1.x()-p0.x())*(p2.x()-p0.x()))/p2.y())>maxErr) {
          for(int i=clusters->containerCount()-1; i>=0; i--) {
            Cluster::Node * child=static_cast<Cluster::Node *>(clusters->containerAt(i));
            validSegments(child, maxErr, curve);
          }
          return;
        }
      }
      if(App::verbosity()>=1) {
        App::log(tr("Found valid segment %1\n").arg(clusters->toString(80)));
      }
      _validClusters.append(indexList);
    }
  }

  /*!
    \a x is the unsorted list of x values
  */
  void CurveSplitter::continuousSegments(const VectorList<double>& x)
  {
    return;
    // Get a map of possible x values
    VectorList<double> xSort(x);
    std::sort(xSort.begin(), xSort.end());
    unique(xSort);
    QMap<double, int> xMap;
    for(int i=xSort.count()-1; i>=0; i--) {
      xMap.insert(xSort.at(i), i);
    }

    _validClusters.clear();
    continuousSegments(_clusters, x, xMap);
  }

  void CurveSplitter::continuousSegments(Cluster::Node * clusters,
                                         const VectorList<double>& x,
                                         const QMap<double, int>& xMap)
  {
    VectorList<int> indexList=clusters->indexList();
    int n=indexList.count();
    if(n>1) {  // Get rid of small clusters
      std::sort(indexList.begin(), indexList.end());
      int i1=xMap[x.at(indexList.at(0))];
      for(int j=1; j<n; j++) {
        int i2=xMap[x.at(indexList.at(j))];
        if(i1+1!=i2) {
          for(int i=clusters->containerCount()-1; i>=0; i--) {
            Cluster::Node * child=static_cast<Cluster::Node *>(clusters->containerAt(i));
            continuousSegments(child, x, xMap);
          }
          return;
        }
        i1=i2;
      }
      _validClusters.append(indexList);
    }
  }

} // namespace QGpCoreMath

