/***************************************************************************
**
**  This file is part of GeopsyGui.
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
**  License for more details.
**
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2007-02-02
**  Copyright: 2007-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <QtGui>
#include <GeopsyCore.h>
#include "SignalFileItem.h"

namespace GeopsyGui {

/*!
  \class SignalFileItem SignalFileItem.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
SignalFileItem::SignalFileItem(SignalDatabase * db, QObject * parent)
    : QAbstractItemModel(parent)
{
  TRACE;
  _db=db;
}

/*!
  \fn SignalFileItem::~SignalFileItem()
  Description of destructor still missing
*/

void SignalFileItem::beginReset()
{
  TRACE;
  beginResetModel();
}

void SignalFileItem::endReset()
{
  TRACE;
  endResetModel();
}

void SignalFileItem::beginAddFile()
{
  TRACE;
  int n=_db->filePool().count();
  beginInsertRows(QModelIndex(), n, n);
}

void SignalFileItem::endAddFile()
{
  TRACE;
  endInsertRows();
}

int SignalFileItem::rowCount(const QModelIndex & parent) const
{
  TRACE;
  if(!parent.isValid()) {
    return _db->filePool().count();
  } else {
    return 0;
  }
}

/*!
  Currently return only 1 but may evolve in the future
*/
int SignalFileItem::columnCount(const QModelIndex &) const
{
  TRACE;
  return 2;
}

QVariant SignalFileItem::data(const QModelIndex &index, int role) const
{
  TRACE;
  if(!index.isValid()) return QVariant();
  SignalFile * f=file(index);
  switch (role) {
  case Qt::DisplayRole:
    if(index.column()==0) {
      return f->shortName();
    } else {
      return f->name();
    }
  case Qt::ToolTipRole:
    return f->name();
  default:
    return QVariant();
  }
}

QModelIndex SignalFileItem::parent (const QModelIndex & ) const
{
  TRACE;
  return QModelIndex();
}

QModelIndex SignalFileItem::index (int row, int column, const QModelIndex &) const
{
  TRACE;
  // 2019-01-21
  // One of the strange behavior under Windows, count() is apparently not called
  // and this function is used with row==0 and an empty file pool
  // 2019-11-11
  // Happened also under CentOS 7 with Qt 5.12.6
  // Hence add this test for all platform, it does not hurt
  SignalFilePool& files=_db->filePool();
  if(row<files.count()) {
   return createIndex(row, column, files.at(row));
  } else {
    return QModelIndex();
  }
}

SignalFile * SignalFileItem::file(const QModelIndex& index)
{
  TRACE;
  return static_cast<SignalFile *>(index.internalPointer());
}

QModelIndex SignalFileItem::index(SignalFile * f)
{
  TRACE;
  if(f) {
    int i=_db->filePool().indexOf(f);
    if(i>-1) return createIndex(i, 0, f);
  }
  return QModelIndex();
}

/*!
  Loop over all files to provide a sub pool with files sorted in the same way as displayed in the
  file list. Using QItemSelectionModel::selectedRows() leads to random order (probably a QHash somewhere).
*/
void SignalFileItem::getSelection(SubSignalPool& sel, const QItemSelectionModel& sm)
{
  TRACE;
  static const QModelIndex parent;
  int nFiles=_db->filePool().count();
  for(int i=0;i<nFiles;i++) {
    if(sm.isRowSelected(i, parent)) {
      sel.addFile(_db->filePool().at(i));
    }
  }
}

void SignalFileItem::getSelection(QList<SignalFile *>& sel, const QItemSelectionModel& sm)
{
  TRACE;
  QModelIndexList l=sm.selectedRows();
  for(QModelIndexList::iterator it=l.begin(); it !=l.end(); it++) {
    sel << _db->filePool().at(it->row());
  }
}

void SignalFileItem::remove(const QList<SignalFile *>& sel)
{
  TRACE;
  for(QList<SignalFile *>::const_iterator it=sel.begin();it!=sel.end();++it) {
    QModelIndex i=index(*it);
    beginRemoveRows(QModelIndex(), i.row(), i.row());
    _db->removeFile( *it);
    endRemoveRows();
  }
}

} // namespace GeopsyGui
