/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-06-01
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef SPARSETIMERANGE_H
#define SPARSETIMERANGE_H

#include <QGpCoreTools.h>
#include "TimeRange.h"
#include "GeopsyCoreDLLExport.h"

namespace GeopsyCore {

class GEOPSYCORE_EXPORT SparseTimeRange : private VectorList<TimeRange>
{
  TRANSLATIONS("SparseTimeRange")
public:
  SparseTimeRange();
  SparseTimeRange(const TimeRange& r);
  ~SparseTimeRange();

  void add(const TimeRange& r, bool acceptOverlap=false);
  void add(const SparseTimeRange& r, bool acceptOverlap=false);
  void remove(const SparseTimeRange& r);
  inline void remove(const TimeRange& r);
  void clear();
  void shift(double dt);
  void round(const DateTime& rt, double period);
  void scale(const DateTime& center, double factor);
  void removeGaps(double dt);
  void removeBlocks(double dt);

  SparseTimeRange invert(bool infiniteEnds=false) const;
  SparseTimeRange invert(const TimeRange& r) const;
  inline SparseTimeRange intersection(const TimeRange& r) const;
  SparseTimeRange intersection(const SparseTimeRange& r) const;
  SparseTimeRange hit(const TimeRange& r) const;
  SparseTimeRange hit(const SparseTimeRange& r) const;
  inline bool intersects(const SparseTimeRange& r) const;
  inline bool intersects(const TimeRange& r) const;
  bool contains(const DateTime& t) const;
  VectorList<TimeRange> split(double minLength, double maxLength) const;

  bool isNull() const {return VectorList<TimeRange>::isEmpty();}
  const VectorList<TimeRange>& ranges() const {return *this;}
  inline TimeRange range() const;
  double totalLength() const;

  void testIndex() const;
  static void testIndex(int maxCount);
  void printDebug() const;

  QString toString() const;
  bool fromString(const QString& s);
  bool fromString(const DateTime& ref, const QString& s);
protected:
  int index(const DateTime& t) const;
  SparseTimeRange intersection(int index, const TimeRange& r) const;
  void remove(int index);
  int remove(int& index, const TimeRange& r);
private:
  void append(const TimeRange& r);
  void insert(int before, const TimeRange& r);

  int _n2;
  static const double _epsilon;
};

inline bool SparseTimeRange::intersects(const TimeRange& r) const
{
  SparseTimeRange res=intersection(r);
  return !res.isEmpty();
}

inline bool SparseTimeRange::intersects(const SparseTimeRange& r) const
{
  SparseTimeRange res=intersection(r);
  return !res.isEmpty();
}

inline SparseTimeRange SparseTimeRange::intersection(const TimeRange& r) const
{
  if(isEmpty()) return *this;
  return intersection(index(r.start()), r);
}

inline void SparseTimeRange::remove(const TimeRange& r)
{
  if(!isEmpty()) {
    int i=index(r.start());
    remove(i, r);
  }
}

inline TimeRange SparseTimeRange::range() const
{
  if(isEmpty())
    return TimeRange();
  else
    return TimeRange(first().start(), last().end());
}

} // namespace GeopsyCore

#endif // SPARSETIMERANGE_H
