/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2003-11-10
**  Copyright: 2003-2019
**    Marc Wathelet
**    Marc Wathelet (ULg, Liège, Belgium)
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <math.h>

#include <QGpCoreTools.h>

#include "SignalFilePool.h"
#include "GeopsyCoreEngine.h"
#include "SignalDatabase.h"

namespace GeopsyCore {

SignalFilePool::~SignalFilePool()
{
  TRACE;
  clear();
}

/*!
  Transfer all \a files to this.
  \a files is cleared without deletion of SignalFile
  that are now own by this object.
*/
void SignalFilePool::add(SignalFilePool& files)
{
  TRACE;
  for(const_iterator it=files.begin(); it!=files.end(); ++it) {
    SignalFile * sigFile=*it;
    sigFile->setDatabase(_db);
    append(sigFile);
  }
  // Remove files for the list of files
  files.QList<SignalFile *>::clear();
}

void SignalFilePool::clear()
{
  TRACE;
  qDeleteAll(*this);
  QList<SignalFile *>::clear();
}

void SignalFilePool::remove(SignalFile * f)
{
  TRACE;
  int index=indexOf(f);
  if(index!=-1) {
    QList<SignalFile *>::removeAt(index);
    delete f;
  }
}

bool SignalFilePool::load(QString name, SignalFileFormat format)
{
  TRACE;
  QFileInfo f(name);
  QString pname=f.absoluteFilePath();
  // Checking if file name is unique is no longer usefull with new structure
  // where original signals cannot be modified (their samples)
  // We keep this code for a while, tracking side effects.
  // This is one of the slowest part when many signals are imported.
  // check if the new filename is unique in the list of files
  /*for(iterator it=begin();it!=end();++it) {
    if(pname==(*it)->name()) {
      Message::warning(MSG_ID, tr("Loading signal ..."),
                            tr("File %1 has already been loaded").arg(pname), Message::cancel(), true);
      return false;
    }
  }*/
  if(f.suffix()=="gpy") {
    return _db->open(pname);
  }
  if(format.id()==SignalFileFormat::Unknown) {
    format=SignalFileFormat::fromContent(pname);
  }
  if(format.id()!=SignalFileFormat::Unknown) {
    SignalFile * newFile=new SignalFile(_db, pname, format);
    bool ret=newFile->load();
    if(!ret) {
      removeAt(indexOf(newFile));
      delete newFile;
    }
    return ret;
  } else {
    Message::warning(MSG_ID, tr("Loading signal ..."),
                          tr("Unknown format for %1").arg(pname), Message::cancel(), true);
    return false;
  }
}

SignalFile * SignalFilePool::find(QString fileName) const
{
  TRACE;
  for(const_iterator it=begin(); it!=end(); ++it) {
    SignalFile * sigFile=*it;
    if(sigFile->shortName()==fileName || sigFile->name()==fileName) {
      return sigFile;
    }
  }
  return 0;
}


} // namespace GeopsyCore
