/***************************************************************************
**
**  This file is part of GeopsyCore.
**
**  GeopsyCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  GeopsyCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2016-11-18
**  Copyright: 2016-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SharkHeader.h"
#include "SignalFileFormat.h"

namespace GeopsyCore {

  /*!
    \class SharkHeader SharkHeader.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  SharkHeader::SharkHeader(SignalFileFormat::Format f)
  {
    TRACE;
    format=f;
    channelCount=0;
    sampleCount=0;
    recordDuration=0;
    samplingFrequency=0.0;
    gain=1.0;
    conversionFactor=1.0;
  }

  /*!
    Return false if last line of header is reached.
  */
  bool SharkHeader::parseLine(const QString& line)
  {
    TRACE;
    if(line.size()<4) {
      comments+=line+"\n";
      return true;
    }
    QString ll=line.toLower();
    switch(ll[0].unicode()) {
    case 'a':
      if(ll.startsWith("altitude")) {
        utmPosition.setZ(line.section(":", 1, 1).replace("m", "").trimmed().toDouble());
        return true;
      }
      break;
    case 'c':
      switch(ll[1].unicode()) {
      case 'h':
        if(ll.startsWith("channel number")) {
          channelCount=line.section(":", 1, 1).trimmed().toInt();
          return true;
        }
        break;
      case 'o':
        if(ll.startsWith("conversion factor")) {
          conversionFactor=line.section(":", 1, 1).trimmed().toDouble();
          return true;
        }
        break;
      }
      break;
    case 'e':
      if(ll.startsWith("ending ")) { // Ignore Ending date and time
        return true;
      }
      break;
    case 'g':
      if(ll.startsWith("gain")) {
        gain=line.section(":", 1, 1).trimmed().toDouble();
        return true;
      }
      break;
    case 'm':
      switch(ll[1].unicode()) {
      case 'a':
        if(ll.startsWith("maximum amplitude")) {
          comments+=line+"\n";
          switch(format) {
          case SignalFileFormat::CityShark2:
            return false;
          case SignalFileFormat::MiniShark:
            return true;
          default:
            ASSERT(false);
          }
        }
        break;
      case 'i':
        if(ll.startsWith("minishark version")) {
          miniSharkVersion=Version(line.section(":", 1, 1).trimmed());
          if(miniSharkVersion>Version(1,6)) {
            App::log(tr("Minishark file format version '%1' not fully supported\n")
                             .arg(miniSharkVersion.toString()));
          }
          return true;
        }
        break;
      }
      break;
    case 'l':
      switch(ll[1].unicode()) {
      case 'a':
        if(ll.startsWith("latitude")) {
          switch(format) {
          case SignalFileFormat::CityShark2:
            {
              QString val=line.section(":", 1, 1).trimmed();
              utmPosition.setY((val.section(" ", 0, 0, QString::SectionSkipEmpty).toDouble()+
                                val.section(" ", 1, 1, QString::SectionSkipEmpty).toDouble()/60.0)*
                               (val.section(" ", 2, 2, QString::SectionSkipEmpty)=="N" ? 1.0 : -1.0));
            }
            return true;
          case SignalFileFormat::MiniShark:
            utmPosition.setY(line.section(":", 1, 1).trimmed().toDouble());
            return true;
          default:
            ASSERT(false);
          }
        }
        break;
      case 'o':
        if(ll.startsWith("longitude")) {
          switch(format) {
          case SignalFileFormat::CityShark2:
            {
              QString val=line.section(":", 1, 1).trimmed();
              utmPosition.setX((val.section(" ", 0, 0, QString::SectionSkipEmpty).toDouble()+
                                val.section(" ", 1, 1, QString::SectionSkipEmpty).toDouble()/60.0)*
                               (val.section(" ", 2, 2, QString::SectionSkipEmpty)=="E" ? 1.0 : -1.0));
            }
            return true;
          case SignalFileFormat::MiniShark:
            utmPosition.setX(line.section(":", 1, 1).trimmed().toDouble());
            return true;
          default:
            ASSERT(false);
          }
        }
        break;
      }
      break;
    case 'o':
      if(ll.startsWith("original file name")) {
        originalFileName=line.section(":", 1, 1).trimmed();
        return true;
      }
      break;
    case 'r':
      if(ll.startsWith("record duration") ||
         ll.startsWith("recording duration")) {
        recordDuration=line.section(":", 1, 1).replace("mn", "").trimmed().toInt();
        return true;
      }
      break;
    case 's':
      if(ll.size()<11) {
        comments+=line+"\n";
        return true;
      }
      switch(ll[10].unicode()) {
      case 'a':
        if(ll.startsWith("starting date")) {
          QString d=line.section(":", 1, 1).trimmed();
          switch(format) {
          case SignalFileFormat::CityShark2:
            startTime.setDate(QDate::fromString(d, "dd.MM.yyyy"));
            break;
          case SignalFileFormat::MiniShark:
            startTime.setDate(QDate::fromString(d, "yyyy.MM.dd"));
            break;
          default:
            ASSERT(false);
          }
          return true;
        }
        break;
      case 'i':
        if(ll.startsWith("starting time")) {
          QString t=line.section(":", 1, 3).trimmed();
          startTime.setTime(QTime::fromString(t.section(".", 0, 0), "hh:mm:ss"));
          startTime.addSeconds(qRound(("0."+t.section(".", 1, 1)).toDouble()*1e3)*1e-3);
          return true;
        }
        break;
      case 'e':
        if(ll.startsWith("sample rate")) {
          debugSamplingFrequency=line.section(":", 1, 1);
          samplingFrequency=debugSamplingFrequency.replace(QRegularExpression("[Hh]z"), "").trimmed().toDouble();
          return true;
        }
        break;
      case 'b':
        if(ll.startsWith("sample number")) {
          sampleCount=line.section(":", 1, 1).trimmed().toInt();
          return true;
        }
        break;
      }
    }
    comments+=line+"\n";
    return true;
  }

  bool SharkHeader::isValid(const QFile& f)
  {
    TRACE;
    if (samplingFrequency==0.0) {
      App::log(tr("bad sampling frequency (%1).\n").arg(debugSamplingFrequency) );
      return false;
    }
    if (sampleCount==0) { // Probably Cityshark 1 format
      // Number of samples is not explicitely specified. Recoding duration is used as a proxy
      sampleCount=qRound(recordDuration*60.0*samplingFrequency);
      channelCount=3;
      if(!startTime.isValid()) {
        QFileInfo fi(f);
        startTime=fi.lastModified();
      }
    } else { // Assuming Cityshark 2 or MiniShark format
      // Number of samples is explicitely specified. Recording duration is just a cross-check
      if(round(recordDuration*60.0-sampleCount/samplingFrequency)!=0.0) {
        App::log(tr("recording duration does not match with number of samples and sampling frequency (%1 mn, %2 samples @ %3 Hz).\n")
                      .arg(recordDuration).arg(sampleCount).arg(samplingFrequency));
      }
      if(channelCount==0) {
        App::log(tr("no channel number specified in header, assuming 3.\n") );
        channelCount=3;
      }
    }
    // Convert position lat/long to UTM
    if(utmPosition!=Point()) {
      utmPosition.geographicalToUtm(utmZone);
    }
    return true;
  }

} // namespace GeopsyCore

