/***************************************************************************
**
**  This file is part of DinverDCGui.
**
**  DinverDCGui is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverDCGui is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-05-19
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <DinverDCCore.h>
#include <QGpGuiTools.h>
#include <QGpGuiWave.h>
#include <SciFigs.h>
#include "EllipticityViewer.h"

namespace DinverDCGui {

EllipticityViewer::EllipticityViewer(QWidget * parent)
    : DCModelViewer(parent)
{
  TRACE;
  setObjectName("EllipticityViewer");

  // Tools menu
  QMenu * m;
  QAction * a;

  m=addMenu(tr("&Tools"));

  a=new QAction(tr("Current target"), this);
  a->setCheckable(true);
  a->setChecked(true);
  a->setStatusTip(tr("Show/hide the current target onto the inverted ellipticity curves"));
  connect(a, SIGNAL(toggled(bool)), this, SLOT(showTarget(bool)));
  m->addAction(a);

  _peakTargetLayers=nullptr;
}

EllipticityViewer::~EllipticityViewer()
{
  TRACE;
  delete [] _peakTargetLayers;
}

void EllipticityViewer::setMode(const Mode& m)
{
  TRACE;
  _mode=m;
  setWindowTitle(tr("Ellipticity curves (%1):")
                 .arg(Mode::userValue(m.value())));
}

AbstractLine * EllipticityViewer::targetReferenceLine() const
{
  TRACE;
  ModalLine * line=new ModalLine;
  line->setPen(Pen(Qt::black, 0.6));
  line->setSymbol(Symbol(Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                  Brush(Qt::black, Qt::SolidPattern)));
  return line;
}

void EllipticityViewer::setTitles(int i)
{
  TRACE;
  AxisWindow * w=_graphs[i];
  w->yAxis()->setTitle("Ellipticity angle (deg)");
  w->yAxis()->setScaleType(Scale::Linear);
  w->xAxis()->setFrequency();
  w->setObjectName(QString("ell_%1").arg(i));
  _curveLayers[i]->setSignChangeThreshold(70.0);
}

bool EllipticityViewer::hasCurves(DCReportBlock& dcBlock, int)
{
  TRACE;
  if(dcBlock.ellipticity()) {
    dcBlock.readModalStorages(1);
    return seekMode(dcBlock);
  } else return false;
}

void EllipticityViewer::setSampleCount(DCModelInfo * info, DCReportBlock& dcBlock)
{
  TRACE;
  ModalStorageReader reader(dcBlock.stream());
  int n=reader.nModes();
  if(n>_nGraphs) n=_nGraphs;
  int i, nf;
  for(i=0; i<n; i++) {
    if(reader.seek(dcBlock.stream(),i)) {
      dcBlock.stream() >> nf;
      info->setSampleCount(i,nf);
    } else {
      info->setSampleCount(i,0);
    }
  }
  for(;i<_nGraphs;i++) {
    info->setSampleCount(i,0);
  }
}

void EllipticityViewer::report2plot(DCReportBlock& dcBlock, Point2D ** points, const DCModelInfo& info)
{
  TRACE;
  ModalStorageReader reader(dcBlock.stream());
  int n=reader.nModes();
  if(n>_nGraphs) n=_nGraphs;
  if(_mode.value()==Mode::Absolute) {
    for(int i=0;i<n;i++) {
      ModalStorageReader::toAbsolutePlot(dcBlock.stream(), points[i]+info.offset(i));
    }
  } else {
    for(int i=0;i<n;i++) {
      ModalStorageReader::toPlot(dcBlock.stream(), points[i]+info.offset(i));
    }
  }
}

void EllipticityViewer::setLimits(int i, const Rect& r)
{
  TRACE;
  _graphs[i]->xAxis()->setRange(r.x1()*0.95, r.x2()*1.05);
  if(_mode.value()==Mode::Absolute) {
    _graphs[i]->yAxis()->setRange(0.0, 90.0);
  } else {
    _graphs[i]->yAxis()->setRange(-90.0, 90.0);
  }
}

bool EllipticityViewer::seekMode(DCReportBlock& dcBlock)
{
  TRACE;
  return dcBlock.modalStorage(0);
}

void EllipticityViewer::setTarget(const TargetList * tl)
{
  TRACE;
  const QList<ModalCurve>& curves=tl->ellipticityCurveTarget().curves();
  for(int i=0; i<_nGraphs; i++) {
    _mode.setIndex(i);
    for(auto it=curves.begin(); it!=curves.end(); it++) {
      if(it->isEnabled()) {
        if(it->hasMode(_mode)) {
          static_cast<ModalLine *>(_targetLayers[i]->addLine())->setCurve(*it);
        } else {
          App::log(tr("Curve '%1' does not have mode '%2'\n")
                   .arg(it->name())
                   .arg(_mode.toString(Mode::Ellipticity)));
        }
      }
    }
    if(i==0)
      addText(i, tr("Fundamental mode"));
    else
      addText(i, tr("Higher mode %1").arg(i));
  }
  // Peak
  _peakTargetLayers=new ParallelBands *[_nGraphs];
  for(int i=0; i<_nGraphs; i++) {
    _peakTargetLayers[i]=new ParallelBands(_targetLayers[i]->graph());
    _peakTargetLayers[i]->setOpacity(0);
  }
  const EllipticityPeakTarget& ellTarget=tl->ellipticityPeakTarget();
  if(ellTarget.selected() && ellTarget.value().isValid()) {
    double f0=ellTarget.value().mean();
    double df0=ellTarget.value().stddev();
    for(int i=0; i<_nGraphs; i++) {
      _peakTargetLayers[i]->setOpacity(0.3);
      _peakTargetLayers[i]->addBand(f0-df0, f0, QBrush(Qt::lightGray));
      _peakTargetLayers[i]->addBand(f0+df0, f0, QBrush(Qt::gray));
    }
  }
}

void EllipticityViewer::showTarget(bool isVisible)
{
  TRACE;
  DCModelViewer::showTarget(isVisible);
  for(int i=0;i<_nGraphs;i++) {
    _peakTargetLayers[i]->setOpacity(isVisible ? 0.3 : 0.0);
    _peakTargetLayers[i]->deepUpdate();
  }
}


} // namespace DinverDCGui
