/***************************************************************************
**
**  This file is part of DinverDCGui.
**
**  DinverDCGui is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverDCGui is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2006-05-19
**  Copyright: 2006-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include <DinverDCCore.h>
#include <QGpCoreWave.h>
#include <QGpGuiTools.h>
#include <QGpGuiWave.h>
#include <SciFigs.h>
#include "AutocorrViewer.h"

namespace DinverDCGui {

AutocorrViewer::AutocorrViewer(QWidget * parent)
    : DCModelViewer(parent)
{
  TRACE;
  setObjectName("AutocorrViewer");
  _versusDistance=false;

  // Tools menu
  QMenu * m;
  QAction * a;

  m=addMenu(tr( "&Tools" ));

  a=new QAction(tr( "Current target" ), this);
  a->setCheckable(true);
  a->setChecked(true);
  a->setStatusTip(tr( "Show/hide the current target onto the inverted autocorrelation curves" ));
  connect(a, SIGNAL(toggled(bool)), this, SLOT(showTarget(bool)) );
  m->addAction(a);
}

AutocorrViewer::~AutocorrViewer()
{
  TRACE;
}

AbstractLine * AutocorrViewer::targetReferenceLine() const
{
  TRACE;
  ModalLine * line=new ModalLine;
  line->setPen(Pen( Qt::black, 0.6) );
  line->setSymbol(Symbol( Symbol::Circle, 1.2, Pen(Qt::black, 0.0),
                              Brush(Qt::black, Qt::SolidPattern) ));
  return line;
}

void AutocorrViewer::setMode(const Mode& m)
{
  TRACE;
  _mode=m;
  setWindowTitle(tr("%1 autocorrelation curves (mode %2):")
                 .arg(Mode::convertPolarization(m.polarization())).arg(m.index()));
}

void AutocorrViewer::setTitles(int i)
{
  TRACE;
  AxisWindow * w=_graphs[i];
  w->yAxis()->setRange(-0.5, 1.1);
  w->yAxis()->setTitle("Autocorr ratio");
  w->yAxis()->setTitleInverseScale("1/Autocorr ratio");
  if(_versusDistance) {
    w->xAxis()->setTitle("Distance (m)");
    w->xAxis()->setTitleInverseScale("1/distance (1/m)");
  } else {
    w->xAxis()->setFrequency();
  }
  w->setObjectName(QString("spac_%1").arg(i));
}

bool AutocorrViewer::hasCurves(DCReportBlock& dcBlock, int)
{
  TRACE;
  return dcBlock.autocorr();
}

inline int AutocorrViewer::polarizationIndex()
{
  TRACE;
  switch (_mode.polarization()) {
  case Mode::Radial:
    return 1;
  case Mode::Transverse:
    return 2;
  default:
    break;
  }
  return 0;
}

/*!
  hasCurve() must be called just before this function because internal _nRings must be correclty initialized
*/
void AutocorrViewer::setSampleCount(DCModelInfo * info, DCReportBlock& dcBlock)
{
  TRACE;
  int nRings;
  dcBlock.stream() >> nRings;
  dcBlock.readModalStorages(nRings*3);
  if(nRings>_nGraphs) nRings=_nGraphs;
  int pi=polarizationIndex();
  int nf;
  for(int i=0;i<nRings;i++) {
    if(dcBlock.modalStorage(3*i+pi)) {
      ModalStorageReader reader(dcBlock.stream());
      if(reader.seek(dcBlock.stream(),_mode.index())) {
        dcBlock.stream() >> nf;
        info->setSampleCount(i,nf);
        continue;
      }
    }
    info->setSampleCount(i,0);
  }
  // For the sake of security, set 0 samples to remaining graphs
  for(int i=nRings;i<_nGraphs;i++) {
    info->setSampleCount(i,0);
  }
}

void AutocorrViewer::report2plot(DCReportBlock& dcBlock, Point2D ** points, const DCModelInfo& info)
{
  TRACE;
  int nRings;
  dcBlock.stream() >> nRings;
  dcBlock.readModalStorages(nRings*3);
  if(nRings>_nGraphs) nRings=_nGraphs;
  int pi=polarizationIndex();
  for(int i=0;i<nRings;i++) {
    if(dcBlock.modalStorage(3*i+pi)) {
      ModalStorageReader reader(dcBlock.stream());
      if(reader.seek(dcBlock.stream(),_mode.index())) {
        reader.toPlot(dcBlock.stream(), points[i]+info.offset(i));
      }
    }
  }
}

void AutocorrViewer::setLimits(int i, const Rect& r)
{
  TRACE;
  _graphs[ i ] ->xAxis()->setRange(r.x1() * 0.95, r.x2() * 1.05);
  _graphs[ i ] ->yAxis()->setRange( -0.5, 1.1);
}

void AutocorrViewer::setTarget(const TargetList * tl)
{
  TRACE;
  const AutocorrCurves& ac=tl->autocorrTarget().curves();
  if(ac.isEmpty()) return;
  int nRings=ac.ringCount();
  if(nRings>_nGraphs) nRings=_nGraphs;
  for(int i=0;i<nRings;i++) {
    _mode.setRingIndex(i);
    const QList<ModalCurve>& curves=ac.ringCurves(i);
    for(QList<ModalCurve>::const_iterator it=curves.begin();it!=curves.end(); it++) {
      if(it->hasMode(_mode)) {
        static_cast<ModalLine *>(_targetLayers[i]->addLine())->setCurve( *it);
      }
    }
    const AutocorrRing& r=ac.ring(i);
    addText(i, tr("Ring %1 %2").arg(i+1).arg(r.toUserString()));
  }
}

} // namespace DinverDCGui
