/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-07-09
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ArtificialBeeColony.h"
#include "AbstractForward.h"
#include "ModelSet.h"

namespace DinverCore {

  /*!
    \class ArtificialBeeColony ArtificialBeeColony.h
    \brief Brief description of class still missing

    Built from Karaboga and Basturk (2007)
  */

  /*!
    Description of constructor still missing
  */
  ArtificialBeeColony::ArtificialBeeColony(int modelCount,
                                           int giveUp,
                                           int seed)
  {
    TRACE;
    _forward=nullptr;
    _modelCount=modelCount;
    _giveUp=giveUp;
    _parameters=nullptr;
    _misfits=nullptr;
    _lastImproved=nullptr;
    _randomNumbers=new Random(seed);
  }

  /*!
    Description of destructor still missing
  */
  ArtificialBeeColony::~ArtificialBeeColony()
  {
    TRACE;
    delete _forward;
    delete [] _parameters;
    delete [] _misfits;
    delete [] _lastImproved;
    delete _randomNumbers;
  }

  /*!
    Same as ModelRepository::setForward()
  */
  bool ArtificialBeeColony::setForward(const AbstractForward * f)
  {
    TRACE;
    _forward=f->clone();
    _forward->parameterSpace().setVariableParameters();
    if(_forward->parameterSpace().variableParameterCount()<=0) {
      App::log(tr("No variable parameters, check parameterization.\n") );
      return false;
    }
    if(!_forward->parameterSpace().adjustRanges()) {
      return false;
    }
    _forward->parameterSpace().humanInfo();
    return true;
  }

  void ArtificialBeeColony::setStorage()
  {
    delete [] _parameters;
    delete [] _misfits;

    _parameterSpaceChecksum=_forward->parameterSpace().checksum();
    _parameterCount=_forward->parameterSpace().variableParameterCount();
    _parameters=new int[_parameterCount*_modelCount];
    _targetCount=1;
    _misfits=new double[_targetCount*_modelCount];
    _lastImproved=new int[_modelCount];
  }

  /*!
    Init random models
  */
  bool ArtificialBeeColony::init()
  {
    RealSpace& parameterSpace=_forward->parameterSpace();
    if(!_forward->firstModel(*_randomNumbers, _terminated)) {
      return false;
    }
    for(int i=0; i<_modelCount; i++) {
      if(_forward->randomModel(*_randomNumbers, 2)) {
        bool ok=true;
        double m=_forward->misfit(ok);
        if(ok) {
          int * v=_parameters+i*_parameterCount;
          for(int ip=0; ip<_parameterCount; ip++) {
            v[ip]=parameterSpace.variableParameter(ip)->gridValue();
          }
          _misfits[i]=m;
        } else {
          i--;
        }
      } else {
        return false;
      }
    }
    for(int i=0; i<_modelCount; i++) {
      _lastImproved[i]=0;
    }
    _cycle=0;
    return true;
  }

  void ArtificialBeeColony::run(int cycleCount)
  {
    for(int i=0; i<cycleCount; i++) {
      employedBees();
      onlookerBees();
      scoutBees();
      _cycle++;
    }
  }

  void ArtificialBeeColony::improveModel(int index)
  {
    RealSpace& parameterSpace=_forward->parameterSpace();
    int minp, maxp;
    SAFE_UNINITIALIZED(minp, 0);
    SAFE_UNINITIALIZED(maxp, 0);

    int * v=_parameters+index*_parameterCount;
    for(int ip=0; ip<_parameterCount; ip++) {
      parameterSpace.variableParameter(ip)->setGridValue(v[ip]);
    }
    int j=_randomNumbers->uniform(0, _parameterCount-1);
    Parameter * p=parameterSpace.variableParameter(j);
    p->getGridLimits(minp, maxp);
    int newValue;
    do {
      double phi=_randomNumbers->uniform(-1.0, 1.0);
      int k=_randomNumbers->uniform(0, _modelCount-1);
      if(k>=index) {
        k++;
      }
      // Modification in both directions and with a magnitude that depends upon
      // the distance between v and k models along axis j
      newValue=qRound(v[j]+phi*(v[j]-_parameters[k*_parameterCount]));
    } while(newValue<minp && newValue>minp);
    p->setGridValue(newValue);
    bool ok=true;
    double m=_forward->misfit(ok);
    if(ok && m<_misfits[index]) {  // Get a better misfit
      v[j]=newValue;
      _misfits[index]=m;
      _lastImproved[index]=_cycle;

    }
  }

  /*!
    The number of employed bees is the same as the number of models
  */
  void ArtificialBeeColony::employedBees()
  {
    for(int i=0; i<_modelCount; i++) {
      improveModel(i);
    }
  }

  /*!
    The number of onlooker bees is the same as the number of employed bees
  */
  void ArtificialBeeColony::onlookerBees()
  {
    double sum=0.0;
    for(int i=0; i<_modelCount; i++) {
      sum+=_misfits[i];
    }
    for(int i=0; i<_modelCount; i++) {
      double p=_randomNumbers->uniform(0.0, sum);
      int j=0;
      double s=_misfits[j];
      while(p>s) {
        j++;
        s+=_misfits[j];
      }
      improveModel(j);
    }
  }

  /*!
    Search for unimproved models, replace them by a new random model.
  */
  void ArtificialBeeColony::scoutBees()
  {
    RealSpace& parameterSpace=_forward->parameterSpace();
    int thresholdCycle=_cycle-_giveUp;
    int i=0;
    while(i<_modelCount) {
      if(_lastImproved[i]<thresholdCycle) {
        if(_forward->randomModel(*_randomNumbers, 2)) {
          bool ok=true;
          double m=_forward->misfit(ok);
          if(ok) {
            int * v=_parameters+i*_parameterCount;
            for(int ip=0; ip<_parameterCount; ip++) {
              v[ip]=parameterSpace.variableParameter(ip)->gridValue();
            }
            _misfits[i]=m;
            i++;
          }
        }
      }
    }
  }

} // namespace DinverCore

