/***************************************************************************
**
**  This file is part of DinverCore.
**
**  DinverCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  DinverCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2009-03-23
**  Copyright: 2009-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef ACTIVEMODEL_H
#define ACTIVEMODEL_H

#include <QGpCoreTools.h>

#include "DinverCoreDLLExport.h"
#include "Index.h"

namespace DinverCore {

  class DINVERCORE_EXPORT ActiveModel
  {
    TRANSLATIONS("ActiveModel")
  public:
    inline void init(const SetIndex& i, double parentVolume);
    inline void operator=(const ActiveModel& o);

    const SetIndex& index() const {return _index;}

    int acceptedCount() const {return _nAccepted;}
    void addAccepted() {_nAccepted++;}

    int rejectedCount() const {return _nRejected;}
    void addRejected() {_nRejected++;}

    void addLimit() {_nLimit++;}
    int limitCount() const {return _nLimit;}

    bool isAtomic() const {return _atomic;}
    void setAtomic() {_atomic=true;}

    inline bool giveUp(double ratio) const;

    int navigatorHitCount() const {return _navigatorHitCount;}
    inline void addNavigatorHit();
    int navigatorHitTime() const {return _navigatorHitTime;}
    void setNavigatorHitTime(int n=0) {_navigatorHitTime=n;}

    bool isDying() const {return false;}
    bool isAlive() const {return true;}
    inline void healthCheck();

    void setVolume(double v) {_volume=v;}
    double volume() const {return _volume;}

    static inline bool compareHitCount(const ActiveModel * m1, const ActiveModel * m2);
  private:
    SetIndex _index;
    int _nAccepted;
    int _nRejected;
    int _nLimit;
    int _navigatorHitCount;
    int _navigatorHitTime;
    bool _atomic;
    double _volume;
  };

  inline void ActiveModel::init(const SetIndex& i, double parentVolume)
  {
    _index=i;
    _nAccepted=0;
    _nRejected=0;
    _navigatorHitCount=1;         // Hit at least once when created
    _navigatorHitTime=i.value();  // Time of birth is it own index
    _atomic=false;
    _volume=parentVolume;
  }

  inline void ActiveModel::operator=(const ActiveModel& o)
  {
    _index=o._index;
    _nAccepted=o._nAccepted;
    _nRejected=o._nRejected;
    _atomic=o._atomic;
    _navigatorHitCount=o._navigatorHitCount;
    _navigatorHitTime=o._navigatorHitTime;
    _volume=o._volume;
  }

  inline void ActiveModel::healthCheck()
  {
    // Negative navigator hit count are always considered as alive.
    /*if(_nNavigatorHits>0) {
      _nNavigatorHits=0;
      _lives++;
    } else if(_nNavigatorHits==0) {
       if(isAlive()) _lives--;
    }*/
  }

  inline void ActiveModel::addNavigatorHit()
  {
    /*if(!alive()) {
      App::log(tr("Back to life\n") );
      _lives++;
    }*/
    _navigatorHitCount++;
  }

  inline bool ActiveModel::giveUp(double ratio) const
  {
    return _nRejected>50 &&
           _nRejected>ratio*(double)(_nRejected+_nAccepted);
  }

  inline bool ActiveModel::compareHitCount(const ActiveModel * m1, const ActiveModel * m2)
  {
    return m1->navigatorHitCount()>m2->navigatorHitCount();
  }

} // namespace DinverCore

#endif // ACTIVEMODEL_H
