/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-08-29
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#ifndef WAVEFIELD_H
#define WAVEFIELD_H

#include <QGpCoreMath.h>

#include "ArrayCoreDLLExport.h"
#include "AbstractFKFunction.h"
#include "WavefieldValues.h"

namespace ArrayCore {

  class ARRAYCORE_EXPORT Wavefield
  {
  public:
    Wavefield();
    ~Wavefield();

    bool operator==(const Wavefield& o) const;

    void setBlockCount(int blockCount);
    void setSensors(const VectorList<Point2D>& pos);
    void addWave(double amplitude, double wavenumber,
                 double theta, double xi);
    void clearWaves() {_waves.clear();}
    void setAmplitude(int i, double amplitude);
    void setSignalPower();
    void setEllipticity(int i, double xi);
    void setIncoherentNoise(double R, double sigma);
    void setParameters(const Vector<double>& p);
    void getParameters(Vector<double>& p) const;
    void setMaximumShift(Vector<double>& maxShift) const;
    void setPrecision(Vector<double>& prec) const;

    int waveCount() const {return _waves.count();}
    int blockCount() const {return _blockCount;}
    int sensorCount() const {return _sensors.count();}

    double wavenumber(int index) const {return _waves.at(index).k;}
    double theta(int index) const {return _waves.at(index).theta;}
    double amplitude(int index) const {return _waves.at(index).a;}
    double ellipticity(int index) const {return _waves.at(index).xi;}
    double incoherentNoiseRation() const;
    double sigma() const {return _etaH/_etaZ;}

    Point2D sensor(int index) const;
    const FKSteering * steering() const;
    VectorList<Point2D> sensors() const;

    Complex east(int iw, int ib) const;
    Complex north(int iw, int ib) const;
    Complex vertical(int iw, int ib) const;

    void setFieldVectors();
    ComplexMatrix crossSpectrum() const;
    ComplexMatrix crossSpectrumDerivative(int parameterIndex) const;

    void initMisfit(const VectorList<WavefieldValues::Observations>& obs, const FKSteering * steering);
    double misfit(const VectorList<WavefieldValues::Observations>& obs);
    double misfitDerivative(int parameterIndex, const VectorList<WavefieldValues::Observations>& obs);
    QString detailedMisfit(const VectorList<WavefieldValues::Observations>& obs);
  private:
    void allocateFieldVectors();
    double positivePenalty(double param) const;
    double positivePenaltyDerivative(double param) const;
    double ellipticityPenalty(double xi, double xih, double xiz) const;
    double ellipticityPenaltyDerivative(double xi, double xih, double xiz) const;

    static GlobalRandom * _phaseGenerator;
    class Wave
    {
    public:
      bool operator==(const Wave& o) const;

      double a, k, theta, xi;
      VectorList<double> phi;
      Complex kvec;
      double ctheta, stheta;
      double cxi, sxi;
    };
    int _blockCount;
    double _etaH, _etaZ, _signalPower;
    VectorList<Wave> _waves;
    VectorList<Complex> _sensors;

    VectorList<Complex> _expCache;
    ComplexMatrix * _fields;

    ComplexMatrix _invFsim;
    VectorList<WavefieldValues> _simValues;
  };

} // namespace ArrayCore

#endif // WAVEFIELD_H

