/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2019-03-07
**  Copyright: 2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "SPACRing.h"

namespace ArrayCore {

  /*!
    \class SPACRing SPACRing.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  SPACRing::SPACRing()
    : AutocorrRing()
  {
    TRACE;
  }

  /*!
    Description of constructor still missing
  */
  SPACRing::SPACRing(double minR, double maxR)
    : AutocorrRing(minR, maxR)
  {
    TRACE;
  }

  /*!
    Description of destructor still missing
  */
  SPACRing::~SPACRing()
  {
    TRACE;
  }

  bool SPACRing::lessThanDistance(const StationPair * p1,
                                  const StationPair * p2)
  {
    return p1->distance()<p2->distance();
  }

  bool SPACRing::lessThanAzimuth(const StationPair * p1,
                                 const StationPair * p2)
  {
    return p1->azimuth()<p2->azimuth();
  }

  /*!
    Add a pair and update max radius.
  */
  void SPACRing::add(const StationPair * p)
  {
    TRACE;
    if(_pairs.isEmpty()) {
      setMinRadius(p->distance());
      setMaxRadius(p->distance());
    } else {
      if(p->distance()>maxRadius()) {
        setMaxRadius(p->distance());
      }
    }
    _pairs.append(p);
  }

  void SPACRing::setPairs(const VectorList<StationPair>& stationPairs)
  {
    TRACE;
    _pairs.clear();
    int nPairs=stationPairs.count(); // Total number of pairs available
    for(int i=0;i<nPairs; i++) {
      const StationPair& p=stationPairs.at(i);
      if(p.distance()>=minRadius() && p.distance()<=maxRadius()) {
        _pairs.append(&p);
      }
    }
  }

  /*!
    Remove first pair (smallest distance, they are ordered)
  */
  void SPACRing::removeFirst()
  {
    _pairs.removeFirst();
    if(_pairs.isEmpty()) {
      setMinRadius(maxRadius());  // Radii are fully set when adding
                                  // the first station.
    } else {
      setMinRadius(_pairs.first()->distance());
    }
  }

  /*!
    Calculate max angle step, eventually without counting the \a startIndex first pairs.
  */
  double SPACRing::maximumAngleStep(int startIndex) const
  {
    TRACE;
    double maxAngle=0.0;
    QList<const StationPair *> pairs=_pairs;
    for(int i=0; i<startIndex; i++) {
      pairs.removeFirst();
    }
    if(pairs.count()<2) {
      return 180.0;
    }
    std::sort(pairs.begin(), pairs.end(), lessThanAzimuth);

    double a;
    for(int i=pairs.count()-2; i>=0; i--) {
      a=pairs.at(i+1)->azimuth().degrees()-pairs.at(i)->azimuth().degrees();
      if(a>maxAngle) {
        maxAngle=a;
      }
    }
    a=pairs.first()->azimuth().degrees()-(pairs.last()->azimuth().degrees()-180.0);
    if(a>maxAngle) {
      maxAngle=a;
    }
    return maxAngle;
  }

  /*!
    Return the maximum relative overlap between two rings.
  */
  double SPACRing::overlap(const SPACRing& o) const
  {
    TRACE;
    if(minRadius()<o.minRadius()) {
      if(o.minRadius()<maxRadius()) {
        if(o.maxRadius()<maxRadius()) {
          return 1.0;
        } else {
          if(thickness()<o.thickness()) {
            return (maxRadius()-o.minRadius())/thickness();
          } else {
            return (maxRadius()-o.minRadius())/o.thickness();
          }
        }
      } else {
        return 0.0;
      }
    } else {
      if(minRadius()<o.maxRadius()) {
        if(maxRadius()<o.maxRadius()) {
          return 1.0;
        } else {
          if(thickness()<o.thickness()) {
            return (o.maxRadius()-minRadius())/thickness();
          } else {
            return (o.maxRadius()-minRadius())/o.thickness();
          }
        }
      } else {
        return 0.0;
      }
    }
  }

  /*!
    Pairs are ordered by increasing distance.
    They are added one by one checking the angle step.
  */
  VectorList<SPACRing> SPACRing::autoRings(const VectorList<StationPair>& pairs,
                                        double maxAngleStep,
                                        double maxRelThickness,
                                        int minPairCount,
                                        double maxOverlap)
  {
    TRACE;
    ASSERT(minPairCount>=1);
    bool line=isLinear(pairs);
    VectorList<SPACRing> rings;
    int n=pairs.count();
    if(n>=2) {
      SPACRing ring;
      ring.add(&pairs.first());
      ring.setMinRadius(pairs.first().distance());
      ring.setMaxRadius(ring.minRadius());
      for(int i=1; i<n; i++) {
        const StationPair& p=pairs.at(i);
        App::log(1, tr("New pair at radius %1 azimuth %2 deg\n")
                 .arg(p.distance()).arg(p.azimuth().degrees()));
        if(p.distance()==ring.maxRadius()) {
          ring.add(&p);
        } else {
          ring.add(&p);
          App::log(1, tr("Enlarging ring...\n"));
          double ringMaxAngleStep=ring.maximumAngleStep();
          App::log(1, tr("  Max angle step %1\n").arg(ringMaxAngleStep));
          if(line || ringMaxAngleStep<=maxAngleStep) { // Yeah, found at least a collection of pairs
                                                       // satifying the maximum angle step condition
            // Try to remove as many pairs as possible still satifying the angle step condition
            if(!line) {
              App::log(tr("Found a ring with angle step %1 (<=%2) %3\n")
                       .arg(ringMaxAngleStep)
                       .arg(maxAngleStep)
                       .arg(ring.toUserString()));
              while(ring.maximumAngleStep(1)<=maxAngleStep && !ring._pairs.isEmpty()) {
                App::log(tr("    remove first pair, condition still satisfied\n"));
                ring.removeFirst();
              }
            }
            App::log(tr("Checking number of pairs (%1>=%2?)\n")
                     .arg(ring.count())
                     .arg(minPairCount));
            // If the ring is sufficiently thin and contains enough pairs, accept it
            if(ring.count()>=minPairCount) {
              App::log(tr("Checking that thickness is at least %1% of the average radius (%2<=%3?)\n")
                       .arg(maxRelThickness*100.0)
                       .arg(ring.thickness())
                       .arg(maxRelThickness*ring.averageRadius()));
              if(ring.thickness()<=maxRelThickness*ring.averageRadius()) {
                App::log(tr("Ring definitively accepted\n"));
                ring.round(0.01);
                rings.append(ring);
                // Just one pair is removed to geneate further rings
                double min=ring.minRadius();
                do {
                  ring.removeFirst();
                } while(ring.minRadius()==min && !ring._pairs.isEmpty());
              } else {
                App::log(tr("Ring rejected\n"));
                /* To satisfy the maximum relative thickness condition, the minimum
                   radius must be increased to a new min given by
                     max-min=err*(max+min)/2
                   Hence,
                     min=(2-err)/(2+err)*max
                */
                double min=(2.0-maxRelThickness)/(2.0+maxRelThickness)*ring.maxRadius();
                do {
                  App::log(tr("  Remove first pair\n"));
                  ring.removeFirst();
                } while(ring.minRadius()<min && !ring._pairs.isEmpty());
              }
              App::log(tr("Starting a new ring %1\n")
                       .arg(ring.toUserString()));
            }
          }
        }
      }
    }
    App::log(tr("Checking overlaps among the %1 rings (max overlap=%2%)\n")
             .arg(rings.count())
             .arg(maxOverlap*100.0));
    // Check overlap
    for(int i=1; i<rings.count(); i++) {
      if(rings.at(i).overlap(rings.at(i-1))>maxOverlap) {
        if(rings.at(i).thickness()<rings.at(i-1).thickness()) {
          rings.removeAt(i-1);
          i--;
        } else {
          rings.removeAt(i);
        }
      }
    }
    return rings;
  }

  /*!
    Returns true if all pairs have the same azimuth
  */
  bool SPACRing::isLinear(const VectorList<StationPair>& pairs)
  {
    if(pairs.count()<2) {
      return true;
    }
    int n=pairs.count();
    double az=pairs.first().azimuth().radians();
    for(int i=1; i<n; i++) {
      if(az!=pairs.at(i).azimuth().radians()) {
        return false;
      }
    }
    return true;
  }

} // namespace ArrayCore

