/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-03-04
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#include "SPACCrossSpectrum.h"
#include "StationPair.h"
#include "SPACRing.h"

namespace ArrayCore {

  /*!
    \class SPACCrossSpectrum SPACCrossSpectrum.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  SPACCrossSpectrum::SPACCrossSpectrum(const ArraySelection * array,
                                       const SPACParameters * param)
      : ArrayCrossSpectrum(array, param), _stationPairs(array)
  {
    _horizontalMatrix=nullptr;
    _rings=_stationPairs.rings(parameters());
    resize();
  }

  SPACCrossSpectrum::~SPACCrossSpectrum()
  {
    delete _horizontalMatrix;
  }

  void SPACCrossSpectrum::resize()
  {
    delete _horizontalMatrix;
    _horizontalMatrix=nullptr;
    switch (_mode) {
    case ArrayStations::ThreeComponents:
      _horizontalMatrix=new ComplexMatrix(2*_array.count());
      break;
    case ArrayStations::Horizontal:
      ASSERT(false);
    case ArrayStations::Vertical:
      break;
    }
    delete _matrix;
    _matrix=new ComplexMatrix(_array.count());
  }

  bool SPACCrossSpectrum::calculate(const VectorList<int>& blocks)
  {
    TRACE;
    if(_mode==ArrayStations::ThreeComponents) {
      _horizontalMatrix->zero();
    }
    resetMatrix();

    for(VectorList<int>::const_iterator it=blocks.begin(); it!=blocks.end(); it++) {
      if(lockTimeWindow(_timeWindows->list().at(*it))) {
        if(_mode==ArrayStations::ThreeComponents) {
          addHorizontal(*_horizontalMatrix);
        }
        addVertical();
        unlockTimeWindow();
      }
    }

    if(_mode==ArrayStations::ThreeComponents) {
      (*_horizontalMatrix)*=1.0/static_cast<double>(blocks.count());
      // Normalization is more complex than for vertical
      // TODO??
    }
    meanMatrix(blocks.count());
    normalize(*_matrix);

    return true;
  }

  void SPACCrossSpectrum::normalize(ComplexMatrix& covmat)
  {
    TRACE;
    int n=covmat.rowCount();
    double * diag=new double[n];
    for(int i=0; i<n; i++) {
      const Complex& c=covmat.at(i, i);
      ASSERT(c.im()<c.re()*1e-10);
      diag[i]=1.0/::sqrt(c.re());
    }
    for(int i=0; i<n; i++) {
      int j;
      for(j=0; j<i; j++) {
        covmat.at(i, j)*=diag[i]*diag[j];
      }
      covmat.at(i, i)=1.0;
      for(j++; j<n; j++) {
        covmat.at(i, j)*=diag[i]*diag[j];
      }
    }
    delete [] diag;
  }

  Complex SPACCrossSpectrum::radialAutocorr(int station1, int station2, const Angle& azimuth) const
  {
    int n=_stations.count();
    int iE1=station1;
    int iN1=n+station1;
    int iE2=station2;
    int iN2=n+station2;
    double sin2=azimuth.sin()*azimuth.sin();
    double cos2=azimuth.cos()*azimuth.cos();
    double sincos=azimuth.sin()*azimuth.cos();
    Complex c;
    c+=_horizontalMatrix->at(iN1, iN2)*sin2;
    c+=_horizontalMatrix->at(iN1, iE2)*sincos;
    c+=_horizontalMatrix->at(iE1, iN2)*sincos;
    c+=_horizontalMatrix->at(iE1, iE2)*cos2;
    Complex pow1;
    pow1+=_horizontalMatrix->at(iN1, iN1)*sin2;
    pow1+=(_horizontalMatrix->at(iN1, iE1)+_horizontalMatrix->at(iE1, iN1))*sincos;
    pow1+=_horizontalMatrix->at(iE1, iE1)*cos2;
    Complex pow2;
    pow2+=_horizontalMatrix->at(iN2, iN2)*sin2;
    pow2+=(_horizontalMatrix->at(iN2, iE2)+_horizontalMatrix->at(iE2, iN2))*sincos;
    pow2+=_horizontalMatrix->at(iE2, iE2)*cos2;
    c/=::sqrt(pow1.re()*pow2.re());
    return c;
  }

  Complex SPACCrossSpectrum::transverseAutocorr(int station1, int station2, const Angle& azimuth) const
  {
    int n=_stations.count();
    int iE1=station1;
    int iN1=n+station1;
    int iE2=station2;
    int iN2=n+station2;
    double sin2=azimuth.sin()*azimuth.sin();
    double cos2=azimuth.cos()*azimuth.cos();
    double sincos=azimuth.sin()*azimuth.cos();
    Complex c;
    c+=_horizontalMatrix->at(iN1, iN2)*cos2;
    c-=_horizontalMatrix->at(iN1, iE2)*sincos;
    c-=_horizontalMatrix->at(iE1, iN2)*sincos;
    c+=_horizontalMatrix->at(iE1, iE2)*sin2;
    Complex pow1;
    pow1+=_horizontalMatrix->at(iN1, iN1)*cos2;
    pow1-=(_horizontalMatrix->at(iN1, iE1)+_horizontalMatrix->at(iE1, iN1))*sincos;
    pow1+=_horizontalMatrix->at(iE1, iE1)*sin2;
    Complex pow2;
    pow2+=_horizontalMatrix->at(iN2, iN2)*cos2;
    pow2-=(_horizontalMatrix->at(iN2, iE2)+_horizontalMatrix->at(iE2, iN2))*sincos;
    pow2+=_horizontalMatrix->at(iE2, iE2)*sin2;
    c/=::sqrt(pow1.re()*pow2.re());
    return c;
  }

} // namespace ArrayCore
