/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2020-01-29
**  Copyright: 2020
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKRayleighRadial.h"
#include "FKCrossSpectrum.h"

namespace ArrayCore {

  /*!
    \class HRFKRayleighRadial HRFKRayleighRadial.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  inline void HRFKRayleighRadial::maximumEllipticity() const
  {
    _ellOptimization.maximize(_xi, this, M_PI/8.0);
  }

  double HRFKRayleighRadial::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      oneComponentSteering(index);
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(index));
      maximumEllipticity();
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  double HRFKRayleighRadial::value(Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
      oneComponentSteering(kell);
      if(_crossSpectrum) {
        powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));
      }
      maximumEllipticity();
      canonicalEllipticity(_xi.x);
      kell[2]=_xi.x;
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  void HRFKRayleighRadial::setFunctionValue(Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    const Vector<Complex>& e=FKSteering::twoComponentRayleighRadial(a.ell, powerEngine());
    powerEngine()->setHighResolutionValue(e);
    a.value=powerEngine()->value();
  }

  void HRFKRayleighRadial::setFunctionDerivatives(Attributes& a) const
  {
    double value2=a.value*a.value;
    // Get first derivative
    const Vector<Complex>& de=FKSteering::twoComponentRayleighRadialGradient(a.ell, powerEngine());
    a.slope=-2.0*value2*powerEngine()->ellipticityGradient(de);
    // Get second derivative
    double tan=a.ell.im()/a.ell.re();
    a.concavity=powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(a.slope*a.slope/a.value+tan*a.slope-value2*a.concavity);
  }

  /*!
    Calculates a noise factor from the second derivative of beam power
    and the second derivative of the beam pattern.

    One component steering and cross-spectrum are supposed to be correctly initialized.
  */
  double HRFKRayleighRadial::wavenumberNormalizedConcavity(const Vector<double>& kell) const
  {
    Complex ell;
    ell.setUnitExp(kell[2]);
    const Vector<Complex>& e0=FKSteering::twoComponentRayleighRadial(ell, powerEngine());
    powerEngine()->setHighResolutionValue(e0);
    double p=powerEngine()->value();
    steering()->twoComponentSensorRadialProjections(kell, powerEngine());
    powerEngine()->highResolutionKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
    return powerEngine()->highResolutionKRadialSecondDerivative()/p;
  }

} // namespace ArrayCore

