/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-06-28
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKDirectRayleighSquared.h"
#include "HRDirectEllipticityEngine.h"

namespace ArrayCore {

  /*!
    \class HRFKDirectRayleighSquared HRFKDirectRayleighSquared.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  void HRFKDirectRayleighSquared::initialize(const FKParameters& param)
  {
    HRFKDirectRayleigh::initialize(param);
    _ellEngine=new HRDirectEllipticityEngine(cache(), param);
  }

  inline void HRFKDirectRayleighSquared::maximumEllipticity() const
  {
    _ellOptimization.maximize(_xi1, this, M_PI/8.0);
    canonicalEllipticity(_xi1.x);
    // The two peaks are theoretically at the same absolute value.
    _xi2.x=-_xi1.x;
    _ellOptimization.maximize(_xi2, this, M_PI/8.0);
    // Verification that the two peaks are at the same absolute value and opposite
    double dxi=fabs(_xi1.x+_xi2.x);
    while(dxi>M_PI/2.0) {
      dxi=fabs(dxi-M_PI);
    }
    if(dxi>M_PI/180.0) {
      _ellOptimization.curves<HRFKDirectRayleighSquared, HRFKRayleighSquared::Attributes>(this, -M_PI, M_PI,  M_PI/180.0);
      _ellOptimization.step<HRFKDirectRayleighSquared, HRFKRayleighSquared::Attributes>(this, -M_PI, M_PI,  M_PI/180.0, M_PI/8.0);
      fprintf(stderr, "xi1 and xi2 are too different: %lf and %lf\n", _xi1.x, _xi2.x);
      std::abort();
    }
  }

  double HRFKDirectRayleighSquared::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      steering()->threeComponentRayleigh(index, powerEngine());
      maximumEllipticity();
      if(_xi1.value>_xi2.value) {
        return _xi1.value;
      } else {
        return _xi2.value;
      }
    } else {
      return -1.0;
    }
  }

  double HRFKDirectRayleighSquared::value(Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
      steering()->threeComponentRayleigh(kell, powerEngine());
      maximumEllipticity();
      if(_xi1.value>_xi2.value) {
        kell[2]=_xi1.x;
        return _xi1.value;
      } else {
        kell[2]=_xi2.x;
        return _xi2.value;
      }
    } else {
      return -1.0;
    }
  }

  double HRFKDirectRayleighSquared::valueFixed(const Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
      steering()->threeComponentRayleigh(kell, powerEngine());
      _xi1.x=kell[2];
      setFunctionValue(_xi1);
      return _xi1.value;
    } else {
      return -1.0;
    }
  }

  void HRFKDirectRayleighSquared::setFunctionValue(HRFKRayleighSquared::Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    // Get f=P_{theta_h}
    const Vector<Complex>& e=FKSteering::threeComponentRayleighRadial(a.ell, powerEngine());
    powerEngine()->setHighResolutionValue(e);
    a.valueH=powerEngine()->value();
    a.valueH2=a.valueH*a.valueH;
    a.tan=a.ell.im()/a.ell.re();
    a.tan2=a.tan*a.tan;
    a.value=a.tan2*a.valueH2;
  }

  void HRFKDirectRayleighSquared::setFunctionDerivatives(HRFKRayleighSquared::Attributes& a) const
  {
    // Get first derivative of P_{3_h}
    const Vector<Complex>& de=FKSteering::threeComponentRayleighRadialGradient(a.ell, powerEngine());
    a.slope=-2.0*a.valueH2*powerEngine()->ellipticityGradient(de);
    // Get second derivative of P_{3_h}
    double slope2=a.slope*a.slope;
    a.concavity=powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(slope2/a.valueH+a.tan*a.slope-a.valueH2*a.concavity);
    // Get second derivative of P_{3_s}
    double dtan=1.0/(a.ell.re()*a.ell.re());
    a.concavity=2.0*(dtan*(dtan+2.0*a.tan2)*a.valueH2
                     +4.0*a.tan*dtan*a.valueH*a.slope
                     +a.tan2*(slope2+a.valueH*a.concavity));
    // Get first derivative of P_{3_s}
    a.slope=2.0*(a.tan*dtan*a.valueH2+a.tan2*a.valueH*a.slope);
  }

  void HRFKDirectRayleighSquared::setAssociatedResults(const Vector<double>& kell, double power,
                                                       FKPeaks::Value& val) const
  {
    val.power=sqrt(power)/tan(kell[2]);
    rawEllipticity(kell, val);
  }

} // namespace ArrayCore

