/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-06-28
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKDirectRayleigh.h"
#include "FKCrossSpectrum.h"

namespace ArrayCore {

  /*!
    \class HRFKDirectRayleigh HRFKDirectRayleigh.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

// Get a detailled timing report of the processing
// Run with verbosity level to 3 to get the report.
// Configure with '-D PROCESS_TIMING'
#ifdef PROCESS_TIMING
  static double tinv=0.0;
  static Mutex globalTimeMutex;
#endif

  HRFKDirectRayleigh::HRFKDirectRayleigh(FKCache * cache)
    : ConvFKRayleigh(cache)
  {

  }

  HRFKDirectRayleigh::~HRFKDirectRayleigh()
  {
#ifdef PROCESS_TIMING
    App::log(3, tr("HRFKDirectRayleigh: %1 (ms; mat inv)\n")
              .arg(tinv*1e-6));
#endif
  }

  void HRFKDirectRayleigh::initialize(const FKParameters& param)
  {
    setPowerEngine(new FKPower(cache()->array()->count(), 3));
    if(_crossSpectrum) {
      powerEngine()->setCrossSpectrum(_crossSpectrum->matrix());
    }
    steeringInitialization()->setSensorPositionErrors(param);
    steeringInitialization()->setSensorOrientationErrors(param);
    steeringInitialization()->setPhaseShifts(param.gridStep(), param.effectiveGridSize());
    steeringInitialization()->setRadialRotations(param.gridStep());
    _ellOptimization.setAbsolutePrecision(param.ellipticityAbsolutePrecision());
    AbstractFKRayleigh::initialize(param);
  }

  void HRFKDirectRayleigh::resetCrossSpectrum()
  {
    _crossSpectrum->resetMatrix();
  }

  void HRFKDirectRayleigh::addCrossSpectrum()
  {
    _crossSpectrum->addThreeComponent(steering()->sensorOrientationErrors());
  }

  void HRFKDirectRayleigh::meanCrossSpectrum(int nBlocks)
  {
    _crossSpectrum->meanMatrix(nBlocks);
  }

  bool HRFKDirectRayleigh::invertCrossSpectrum()
  {
#ifdef PROCESS_TIMING
    QElapsedTimer chrono;
    chrono.start();
#endif
    bool ret=_crossSpectrum->invert();
#ifdef PROCESS_TIMING
    qint64 t=chrono.nsecsElapsed();
    globalTimeMutex.lock();
    tinv+=t;
    globalTimeMutex.unlock();
#endif
    return ret;
  }

  void HRFKDirectRayleigh::addIncoherentNoise(double horizontal, double vertical)
  {
    ComplexMatrix& covmat=*_crossSpectrum->matrix();
    int n=_crossSpectrum->array().count();
    int n2=n*2;
    int i1, i2;
    for(int i=0; i<n; i++) {
      covmat.at(i, i)+=horizontal;
      i1=i+n;
      i2=i+n2;
      covmat.at(i1, i1)+=horizontal;
      covmat.at(i2, i2)+=vertical;
    }
  }

  const ComplexMatrix * HRFKDirectRayleigh::crossSpectrum() const
  {
    if(_crossSpectrum) {
      return _crossSpectrum->matrix();
    } else {
      return nullptr;
    }
  }

} // namespace ArrayCore

