/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2021-08-27
**  Copyright: 2021
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "HRFKAngularRayleighEllipticity.h"
#include "FKCrossSpectrum.h"

#if 0

namespace ArrayCore {

  /*!
    \class HRFKAngularRayleighEllipticity HRFKAngularRayleighEllipticity.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  HRFKAngularRayleighEllipticity::HRFKAngularRayleighEllipticity(FKCache * cache)
    : HRFKAngularRayleigh(cache)
  {
    _crossSpectrum=nullptr;
    _xir=std::numeric_limits<double>::quiet_NaN();
    _RzOverN=std::numeric_limits<double>::quiet_NaN();
    _RhOverN=std::numeric_limits<double>::quiet_NaN();
  }

  void HRFKAngularRayleighEllipticity::initialize(const FKParameters& param)
  {
    setPowerEngine(new FKPower(cache()->array(), 2));
    steering()->setSensorPositionErrors(param);
    steering()->setSensorOrientationErrors(param);
    steering()->setPhaseShifts(param.gridStep(), param.effectiveGridSize());
  }

  double HRFKAngularRayleighEllipticity::powerValue(const Vector<double>& k, double xi,
                                                    const ComplexMatrix& crossSpectrum) const
  {
    Complex cell;
    // Minus sign because we want re=cos(xi) and im=-sin(xi)
    cell.setUnitExp(-xi);
    steering()->oneComponent(k, powerEngine());
    const Vector<Complex>& e=FKSteering::twoComponentRayleighAngular(cell, powerEngine());
    powerEngine()->setHighResolutionValue(crossSpectrum, e);
    return powerEngine()->value();
  }

  ComplexMatrix HRFKAngularRayleighEllipticity::noiseShiftedCrossSpectrum(double) const
  {
    return *_crossSpectrum;
  }

  double HRFKAngularRayleighEllipticity::sigmaDk(double dr, double err, double p0, double p90, double& b2errdk) const
  {
    PrivateVector<double> kdk(*_k);
    kdk*=1.0+dr;
    double pdk90=powerValue(kdk, 0.5*M_PI, *_crossSpectrum);
    double pdk0=powerValue(kdk, 0.0, *_crossSpectrum);
    TheoreticalFK b2(cache()->array()->relativePos());
    PrivateVector<double> kerr(*_k);
    kerr*=err;
    double b2err=b2.value(kerr);
    PrivateVector<double> dk(*_k);
    dk*=dr;
    dk+=kerr;
    b2errdk=b2.value(dk);

    return (p90*pdk90*(b2errdk*pdk0-b2err*p0))/(p0*pdk0*(b2errdk*pdk90-b2err*p90));
  }

  double HRFKAngularRayleighEllipticity::singularPoint(double p0, double p90) const
  {
    QTextStream s(stdout);
    s << "dr min, dr max, err min, err max? " << Qt::flush;
    QString rep=File::readLine(true);
    if(!rep.isEmpty()) {
      double drMin=rep.section(" ", 0, 0).toDouble();
      double drMax=rep.section(" ", 1, 1).toDouble();
      double errMin=rep.section(" ", 2, 2).toDouble();
      double errMax=rep.section(" ", 3, 3).toDouble();
      double drStep=0.01*(drMax-drMin);
      double errStep=0.01*(errMax-errMin);
      fprintf(stderr, "dr: from %lf to %lf, err: from %lf to %lf\n", drMin, drMax, errMin, errMax);
      QFile f("/tmp/powergrid");
      f.open(QIODevice::WriteOnly);
      QTextStream out(&f);
      out << "x y val\n";
      for(double dr=drMin; dr<drMax; dr+=drStep) {
        for(double err=errMin; err<errMax; err+=errStep) {
          PrivateVector<double> kdk(*_k);
          kdk*=1.0+dr;
          double pdk90=powerValue(kdk, 0.5*M_PI, *_crossSpectrum);
          double pdk0=powerValue(kdk, 0.0, *_crossSpectrum);
          TheoreticalFK b2(cache()->array()->relativePos());
          PrivateVector<double> kerr(*_k);
          kerr*=err;
          double b2err=b2.value(kerr);
          PrivateVector<double> dk(*_k);
          dk*=dr;
          dk+=kerr;
          double b2errdk=b2.value(dk);

          double t1=b2errdk*pdk0-b2err*p0;
          t1*=t1;
          double t2=b2errdk*pdk90-b2err*p90;
          t2*=t2;
          out << dr << " " << err << " " << t1+t2 << Qt::endl;
        }
      }
    }
    return 0.0;
  }

  /*!
    Experimental sigma from wavenumber perturbations
    Fit an hyperbole (a*x+b)/(x+c) to sigma vs. dk observations assuming that there is no k error.
  */
  double HRFKAngularRayleighEllipticity::sigmaDk(double p0, double p90) const
  {
    QTextStream s(stdout);
    s << "dr min, dr max, err? " << Qt::flush;
    QString rep=File::readLine(true);
    if(!rep.isEmpty()) {
      double drMin=rep.section(" ", 0, 0).toDouble();
      double drMax=rep.section(" ", 1, 1).toDouble();
      double err=rep.section(" ", 2, 2).toDouble();
      double drStep=0.01*(drMax-drMin);
      double b2err;
      QFile f("/tmp/powercurve");
      f.open(QIODevice::WriteOnly);
      QTextStream out(&f);
      for(double dr=drMin; dr<drMax; dr+=drStep) {
        double s=sigmaDk(dr, err, p0, p90, b2err);
        out << dr << " " << b2err << " " << s << Qt::endl;
      }
    }
    s << "sigma? " << Qt::flush;
    rep=File::readLine(true);
    if(!rep.isEmpty()) {
      return rep.section(" ", 0, 0).toDouble();
    }

    return 1.0;
#if 0
    // Hyperbolic fit
    int nSamples=10;
    DoubleMatrix mA(2*nSamples, 3), mB(2*nSamples, 1);
    // Estimate shift error with sigma vs. dk
    double b2err, dr, y;
    for(int i=0; i<nSamples; i++) {
      dr=-0.01*(i+1);
      y=sigmaDk(dr, 0.0, p0, p90, b2err);
      mA.at(i, 0)=dr;
      mA.at(i, 1)=1.0;
      mA.at(i, 2)=-y;
      mB.at(i, 0)=dr*y;
      dr=0.01*(i+1);
      y=sigmaDk(dr, 0.0, p0, p90, b2err);
      mA.at(i+nSamples, 0)=dr;
      mA.at(i+nSamples, 1)=1.0;
      mA.at(i+nSamples, 2)=-y;
      mB.at(i+nSamples, 0)=dr*y;
    }
    DoubleMatrix mX=mA.leastSquare(mB);
    double err=-mX.at(2, 0);
    fprintf(stderr, "err with sigma vs. dk=%lf\n", err);
    printf("# sigma vs. dk\n");
    for(double dr=-10; dr<10; dr+=0.005) {
      double s=sigmaDk(dr, 0.0, p0, p90, b2err);
      printf("  r %10.5lf b2err %10.5lf sigma %10.5lf fit %10.5lf\n", dr, b2err, s, (dr*mX.at(0, 0)+mX.at(1, 0))/(dr+mX.at(2, 0)));
    }
    printf("# sigma vs. dk\n");
    for(double dr=-10; dr<10; dr+=0.005) {
      double s=sigmaDk(dr, err, p0, p90, b2err);
      printf("  r %10.5lf b2err %10.5lf sigma %10.5lf fit %10.5lf\n", dr, b2err, s, (dr*mX.at(0, 0)+mX.at(1, 0))/(dr+0*mX.at(2, 0)));
    }
    // We have the shift error, fit sigma vs. b2
    for(int i=0; i<nSamples; i++) {
      dr=-0.01*(i+1);
      y=sigmaDk(dr, err, p0, p90, b2err);
      mA.at(i+nSamples, 0)=b2err;
      mA.at(i+nSamples, 1)=1.0;
      mA.at(i+nSamples, 2)=-y;
      mB.at(i+nSamples, 0)=b2err*y;
      dr=0.01*(i+1);
      y=sigmaDk(dr, err, p0, p90, b2err);
      mA.at(i+nSamples, 0)=b2err;
      mA.at(i+nSamples, 1)=1.0;
      mA.at(i+nSamples, 2)=-y;
      mB.at(i+nSamples, 0)=b2err*y;
    }
    mX=mA.leastSquare(mB);
    err=-mX.at(2, 0);
    fprintf(stderr, "err with sigma vs. b2=%lf\n", err);
    printf("# sigma vs. b2\n");
    for(double dr=-10; dr<10; dr+=0.005) {
      double s=sigmaDk(dr, err, p0, p90, b2err);
      printf("  r %10.5lf b2err %10.5lf sigma %10.5lf fit %10.5lf\n", dr, b2err, s, (dr*mX.at(0, 0)+mX.at(1, 0))/(dr+mX.at(2, 0)));
    }
    //for(double err=-0.50; err<0.5; err+=0.05) {
      //printf("# err=%lf\n", err);
      //for(double dr=-10; dr<10; dr+=0.005) {
      //  double s=sigmaDk(dr, err, p0, p90, b2err);
      //  printf("  r %10.5lf b2err %10.5lf sigma %10.5lf\n", dr, b2err, s);
      //}
    //}
    // The hyperbole is not centered around 0, the shift is mX.at(2, 0)
    /*double err=-mX.at(2, 0);
    fprintf(stderr, "err %10.5lf sigma inf %10.5lf\n", err, mX.at(0, 0));
    // Fit the corrected hyperbole
    for(int i=0; i<nSamples; i++) {
      double dr=(i+1)*0.01;
      double y=sigmaDk(dr, err, p0, p90, b2err);
      mA.at(i, 0)=dr;
      mA.at(i, 1)=1.0;
      mA.at(i, 2)=-y;
      mB.at(i, 0)=dr*y;
    }
    for(int i=0; i<nSamples; i++) {
      double dr=(i+1)*0.01;
      double y=sigmaDk(dr, err, p0, p90, b2err);
      mA.at(nSamples+i, 0)=dr;
      mA.at(nSamples+i, 1)=1.0;
      mA.at(nSamples+i, 2)=-y;
      mB.at(nSamples+i, 0)=dr*y;
    }
    mX=mA.leastSquare(mB);*/
    /*for(double dr=-1; dr<1; dr+=0.05) {
      double s=sigmaDk(dr, err, p0, p90, b2err);
      printf("  r %10.5lf sigma %10.5lf fit %10.5lf\n", dr, s, (dr*mX.at(0, 0)+mX.at(1, 0))/(dr+mX.at(2, 0)));
    }*/
    fprintf(stderr, "sigma inf %10.5lf\n", mX.at(0, 0));
    fflush(stdout);
    return mX.at(0, 0);
#endif
  }

  /*!
    Experimental sigma from points where |B|^2 is supposed to be null
  */
  double HRFKAngularRayleighEllipticity::sigmaNull(const VectorList<Point2D>& nullArrayResponse) const
  {
    PrivateVector<double> knull(*_k);
    int nullIndex=0;
    Statistics stat;
    while(nullIndex<nullArrayResponse.count()) {
      const Point2D& np=nullArrayResponse.at(nullIndex);
      knull[0]=_k->at(0)+np.x();
      knull[1]=_k->at(1)+np.y();
      double p0=powerValue(knull, 0.0, *_crossSpectrum);
      double p90=powerValue(knull, 0.5*M_PI, *_crossSpectrum);
      double sigma=p90/p0;
      if(sigma>0.0) {
        stat.add(sigma);
      }
      nullIndex++;
    }
    return stat.mean();
  }

  /*!
    Calculate noise (Rz/N and Rh/N) and an ellipticity correction.
  */
  void HRFKAngularRayleighEllipticity::setNoise(double xi, const VectorList<Point2D>& nullArrayResponse)
  {
    // Nearly equivalent to P_z (there is just an additionnal sin^2(xi)r))
    double p0=powerValue(*_k, 0.0, *_crossSpectrum);
    double p90=powerValue(*_k, 0.5*M_PI, *_crossSpectrum);

    double sigma;
    sigma=sigmaNull(nullArrayResponse);
    setCorrectedElliticity(sigma, xi, p0, p90);
  }

  /*!

  */
  void HRFKAngularRayleighEllipticity::setCorrectedElliticity(double sigma, double xi, double p0, double p90)
  {
    // Exprimental R/N as a function of sigma
    p0=1.0/p0;
    double p45=1.0/powerValue(*_k, 0.25*M_PI, *_crossSpectrum);
    p90=1.0/p90;
    double dpp=p90+p0;
    double rho090=(p90-p0)/dpp;
    double rho45=1.0-2.0*p45/dpp;
    double a=(rho45*rho45+rho090*rho090)*(sigma+1.0)*(sigma+1.0)+(sigma-1.0)*(sigma-1.0)-2.0*rho090*(1.0-sigma*sigma);
    double b=2.0*(1.0+2.0*sigma)*((rho45*rho45+rho090*rho090)*(sigma+1)-rho090*(1.0-sigma));
    double c=(rho45*rho45+rho090*rho090-1)*(1+2.0*sigma)*(1+2.0*sigma);
    double discriminent2=sqrt(b*b-4.0*a*c);
    double RN1=(-b+discriminent2)/(2.0*a);
    double RN2=(-b-discriminent2)/(2.0*a);
    //printf("Comparison xi and ratio 45/90 %lf %lf %lg\n", rho45/rho090, tan(2.0*xi), fabs(rho45/rho090-tan(2.0*xi)));

    if(RN2>=0.0 || RN1<0.0) {
      App::log(tr("Ambiguous R/N values or negative R/N\n"));
      _xir=std::numeric_limits<double>::quiet_NaN();
      _RzOverN=std::numeric_limits<double>::quiet_NaN();
      _RhOverN=std::numeric_limits<double>::quiet_NaN();
      return;
    }

    // Corrected ellipticity from sigma and R/N
    double delta=(1.0-sigma)/(1.0+2.0*sigma)*RN1;
    double txi=tan(2.0*xi);
    double txi2=txi*txi;
    double dxi=(1-delta*delta)*txi2+1.0;
    if(dxi<0.0) {
      App::log(tr("Negative discriminent to get xir from sigma and R\n"));
      _xir=std::numeric_limits<double>::quiet_NaN();
      _RzOverN=std::numeric_limits<double>::quiet_NaN();
      _RhOverN=std::numeric_limits<double>::quiet_NaN();
      return;
    }
    dxi=sqrt(dxi);
    a=1.0/(txi2+1.0);
    b=-delta*txi2;
    double xir1=0.5*acos((b+dxi)*a);
    double xir2=0.5*acos((b-dxi)*a);
    if(xi<0.0) {
      xir1=-xir1;
      xir2=-xir2;
    }
    if(fabs(delta)<1.0) {
      double r1=fabs(txi-sin(2.0*xir1)/(cos(2.0*xir1)+delta));
      double r2=fabs(txi-sin(2.0*xir2)/(cos(2.0*xir2)+delta));
      if(r1<r2) {
        _xir=xir1;
      } else {
        _xir=xir2;
      }
      _RzOverN=RN1/(1.0+2.0*sigma);
      _RhOverN=_RzOverN*sigma;
    } else {
      if((delta>0.0 && xi>=-M_PI/4.0 && xi<=M_PI/4.0) ||
         (delta<0.0 && (xi<=-M_PI/4 || xi>=M_PI/4))) {
        if(fabs(xir1-xi)<fabs(xir2-xi)) {
          _xir=xir1;
        } else {
          _xir=xir2;
        }
        _RzOverN=RN1/(1.0+2.0*sigma);
        _RhOverN=_RzOverN*sigma;
      } else {
        App::log(tr("sigma and estimated ellipticity are not compatible\n"));
        _xir=std::numeric_limits<double>::quiet_NaN();
        _RzOverN=std::numeric_limits<double>::quiet_NaN();
        _RhOverN=std::numeric_limits<double>::quiet_NaN();
      }
    }
    //fprintf(stderr, "    R/N %lf %lf sigma %lf delta %lf xir %lf\n", RN1, RN2, sigma, delta, Angle::radiansToDegrees(_xir));
  }

  /*!
    Check that the variation of power versus xi is following the theoretical
    shape of a single source.
  */
  bool HRFKAngularRayleighEllipticity::isStableNoise(double xi, double max) const
  {
    double p, sdxi, normalizedR;
    double dxi=M_PI/1024.0;
    for(int i=0; i<=5; i++) {
      p=powerValue(*_k, xi+dxi, *_crossSpectrum);
      sdxi=::sin(dxi);
      sdxi*=sdxi;
      normalizedR=fabs(sdxi*p/((_p0-p)*_RhOverN)-1.0);
      if(normalizedR>max) {
        return false;
      }
      p=powerValue(*_k, xi-dxi, *_crossSpectrum);
      normalizedR=fabs(sdxi*p/((_p0-p)*_RhOverN)-1.0);
      if(normalizedR>max) {
        return false;
      }
      dxi*=2.0;
    }
    return true;
  }

  double HRFKAngularRayleighEllipticity::maximumNoiseDeviation(double xi) const
  {
    double p, sdxi, normalizedR;
    double dxi=M_PI/1024.0;
    double max=0.0;
    for(int i=0; i<=5; i++) {
      p=powerValue(*_k, xi+dxi, *_crossSpectrum);
      sdxi=::sin(dxi);
      sdxi*=sdxi;
      normalizedR=fabs(sdxi*p/((_p0-p)*_RhOverN)-1.0);
      if(normalizedR>max) {
        max=normalizedR;
      }
      p=powerValue(*_k, xi-dxi, *_crossSpectrum);
      normalizedR=fabs(sdxi*p/((_p0-p)*_RhOverN)-1.0);
      if(normalizedR>max) {
        max=normalizedR;
      }
      dxi*=2.0;
    }
    return max;
  }

  void HRFKAngularRayleighEllipticity::debugNoiseCurve(const QString& fileName, const Vector<double>& kell, double freq) const
  {
    double p, sdxi;
    static QMutex rstatLock;
    rstatLock.lock();
    QFile f(fileName);
    f.open(QIODevice::Append);
    QTextStream s(&f);
    double dxi=-M_PI*0.5;
    double vel=2.0*M_PI*freq/kell.length(0, 1);
    s << freq << " " << vel << " " << Angle::radiansToDegrees(kell[2]);
    for(int i=0; i<10; i++) {
      p=powerValue(*_k, kell[2]+dxi, *_crossSpectrum);
      sdxi=::sin(dxi);
      s << " " << sdxi*sdxi*p/((_p0-p)*_RhOverN)-1.0;
      dxi*=0.5;
    }
    dxi=-2.0*dxi;
    for(int i=0; i<10; i++) {
      p=powerValue(*_k, kell[2]+dxi, *_crossSpectrum);
      sdxi=::sin(dxi);
      s << " " << sdxi*sdxi*p/((_p0-p)*_RhOverN)-1.0;
      dxi*=2.0;
    }
    s << "\n";
    f.close();
    rstatLock.unlock();
  }

  void HRFKAngularDirectRayleighEllipticity::initialize(const FKParameters&)
  {
    setPowerEngine(new FKPower(cache()->array(), 3));
  }

  double HRFKAngularDirectRayleighEllipticity::powerValue(const Vector<double>& k, double xi, const ComplexMatrix& crossSpectrum) const
  {
    Complex cell;
    // Minus sign because we want re=cos(xi) and im=-sin(xi)
    cell.setUnitExp(-xi);
    steering()->threeComponentRayleighAngular(k, powerEngine());
    const Vector<Complex>& e=FKSteering::threeComponentRayleighAngular(cell, powerEngine());
    powerEngine()->setHighResolutionValue(crossSpectrum, e);
    return powerEngine()->value();
  }

  ComplexMatrix HRFKAngularDirectRayleighEllipticity::noiseShiftedCrossSpectrum(double whiteNoise) const
  {
    ComplexMatrix covmat=*_crossSpectrum;
    covmat.invert();
    int n=covmat.rowCount()/3;
    whiteNoise*=covmat.trace(Complex()).re()/(3.0*n);
    for(int i=0; i<n; i++) {
      covmat.at(i, i)=whiteNoise;
      covmat.at(i+n, i+n)=whiteNoise;
    }
    covmat.invert();
    return covmat;
  }

} // namespace ArrayCore
#endif
