/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2008-02-08
**  Copyright: 2008-2019
**    Marc Wathelet
**    Marc Wathelet (LGIT, Grenoble, France)
**
***************************************************************************/

#ifndef FKPARAMETERS_H
#define FKPARAMETERS_H

#include "ArrayParameters.h"

namespace ArrayCore {

  class ARRAYCORE_EXPORT FKParameters: public ArrayParameters
  {
  public:
    FKParameters();

    AbstractParameters * clone() const {return new FKParameters(*this);}

    enum ProcessType {Undefined=0,

                      Conventional,
                      ConventionalRadial,
                      ConventionalTransverse,
                      ConventionalRayleigh,

                      Capon,
                      CaponRadial,
                      CaponTransverse,

                      RTBF,
                      ARTBF,
                      RTBFFixedEll,
                      PoggiVertical,
                      PoggiRadial,
                      Omni,

                      ARDS,
                      LDS3,
                      LDS2,
                      SensorOrientation,

                      ActiveConventional,
                      ActiveCapon,
                      ActiveConventionalRayleigh,
                      ActiveRTBF,
                      RDSSingle};
    ENUM_AS_STRING_DECL(ProcessType)

    static QStringList processTypes();
    ProcessType processType() const {return _processType;}
    void setProcessType(ProcessType p) {_processType=p;}
    bool isActiveProcessType() const;

    enum InversionMethod {RefinedGrid, Gradient};
    ENUM_AS_STRING_DECL(InversionMethod)

    InversionMethod inversionMethod() const {return _inversionMethod;}
    void setInversionMethod(InversionMethod m) {_inversionMethod=m;}

    void setCacheGridStep(double s) {_cacheGridStep=s;}
    double cacheGridStep() const;

    void setGridStep(double s) {_gridStep=s;}
    double gridStep() const;

    void setGridStepFactor(double f) {_gridStepFactor=f;}
    double gridStepFactor() const {return _gridStepFactor;}

    void setGridSize(double s) {_gridSize=s;}
    double gridSize() const;

    void setGridSizeFactor(double f) {_gridSizeFactor=f;}
    double gridSizeFactor() const {return _gridSizeFactor;}

    double effectiveGridSize() const;
    double effectiveGridSize(double frequency) const;

    double kmin() const {return _kmin;}
    void setKmin(double k);

    double kmax() const {return _kmax;}
    void setKmax(double k);

    double minimumSlowness() const {return _smin;}
    void setMinimumSlowness(double s) {_smin=s;}

    double maximumSlowness() const {return _smax;}
    void setMaximumSlowness(double s) {_smax=s;}

    double minimumAzimuth() const {return _minAzimuth;}
    void setMinimumAzimuth(double a) {_minAzimuth=a;}

    double maximumAzimuth() const {return _maxAzimuth;}
    void setMaximumAzimuth(double a) {_maxAzimuth=a;}

    void clearCurveRefine();
    void addCurveRefine(double f, double minSlow, double maxSlow);

    inline bool isRefine() const;
    const Curve<Point2D>& refineMinimumSlowness() const {return _refineSlowMin;}
    const Curve<Point2D>& refineMaximumSlowness() const {return _refineSlowMax;}

    double velocityRelativePrecision() const {return _velocityRelativePrecision;}
    void setVelocityRelativePrecision(double p) {_velocityRelativePrecision=p;}

    double ellipticityAbsolutePrecision() const {return _ellipticityAbsolutePrecision;}
    void setEllipticityAbsolutePrecision(double p) {_ellipticityAbsolutePrecision=p;}

    int maximumPeakCount(int sensorCount) const;
    void setMaximumPeakCount(int m) {_maxPeakCount=m;}

    void setMaximumPeakCountFactor(double f) {_maxPeakCountFactor=f;}
    double maximumPeakCountFactor() const {return _maxPeakCountFactor;}

    double absoluteThreshold() const {return _absoluteThreshold;}
    void setAbsoluteThreshold(double a) {_absoluteThreshold=a;}

    double relativeThreshold() const {return _relativeThreshold;}
    void setRelativeThreshold(double r) {_relativeThreshold=r;}

    bool exportAllFKGrids() const {return _exportAllFKGrids;}
    void setExportAllFKGrids(bool g) {_exportAllFKGrids=g;}

    bool saveUndefinedEllipticities() const {return _saveUndefinedEllipticities;}
    void setSaveUndefinedEllipticities(bool s) {_saveUndefinedEllipticities=s;}

    bool sortResultValues() const {return _sortResultValues;}
    void setSortResultValues(bool s) {_sortResultValues=s;}

    int rotateStepCount() const {return _rotateStepCount;}
    void setRotateStepCount(int c) {_rotateStepCount=c;}

    const QList<QStringList>& arrays() const {return _arrays;}

    double damping() const {return _damping>0.0 ? _damping : 0.0;}
    void setDamping(double d) {_damping=d;}

    enum EllipticityCorrection {None, ThreeComponent};
    ENUM_AS_STRING_DECL(EllipticityCorrection)

    EllipticityCorrection ellipticityCorrection() const {return _ellipticityCorrection;}
    void setEllipticityCorrection(EllipticityCorrection e) {_ellipticityCorrection=e;}

    const QString& fixedEllipticityFileName() const {return _fixedEllipticityFileName;}
    void setFixeElliptiticityFileName(const QString& f) {_fixedEllipticityFileName=f;}

    int sensorOrientationErrorCount() const {return _sensorOrientationErrors.count();}
    double sensorOrientationError(const QString& name) const;
    int sensorPositionErrorCount() const {return _sensorPositionErrors.count();}
    Point2D sensorPositionError(const QString& name) const;

    QString sensorOrientationErrorsToString() const;
    QString sensorPositionErrorsToString() const;

    double sensorOrientationApertureAngle() const {return _sensorOrientationApertureAngle;}
    void setSensorOrientationApertureAngle(double a) {_sensorOrientationApertureAngle=a;}

    double sensorOrientationMaximumRadius() const {return _sensorOrientationMaximumRadius;}
    void setSensorOrientationMaximumRadius(double a) {_sensorOrientationMaximumRadius=a;}

    double sensorOrientationMaximumCorrection() const {return _sensorOrientationMaximumCorrection;}
    void setSensorOrientationMaximumCorrection(double c) {_sensorOrientationMaximumCorrection=c;}

    int sensorOrientationMaximumIterations() const {return _sensorOrientationMaximumIterations;}
    void setSensorOrientationMaximumIterations(int n) {_sensorOrientationMaximumIterations=n;}

    int brightSpots() const {return _brightSpots;}
    void setBrightSpots(int n) {_brightSpots=n;}

    double minimumDistance() const {return _minimumDistance;}
    void setMinimumDistance(double m) {_minimumDistance=m;}

    double maximumDistance() const {return _maximumDistance;}
    void setMaximumDistance(double m) {_maximumDistance=m;}

    double sourceGridStep() const {return _sourceGridStep;}
    void setSourceGridStep(double s) {_sourceGridStep=s;}

    double sourceGridSize() const {return _sourceGridSize;}
    void setSourceGridSize(double s) {_sourceGridSize=s;}

    virtual QString toString(PARAMETERS_TOSTRING_ARGS_DECL) const;
    virtual void collectKeywords(PARAMETERS_COLLECTKEYWORDS_ARGS);

    void setDefaultValues(ArrayStations::Mode m);
    void setDefaultValues();
    static double defaultGridStepFactor(ProcessType p, InversionMethod m);

    QString outputName(const QString& groupName) const;
  protected:
    int keywordCount(PARAMETERS_KEYWORDCOUNT_ARGS) const;
    bool setValue(PARAMETERS_SETVALUE_ARGS);
  private:
    void sensorOrientationErrorsFromString(const QString& values, bool& ok);
    void sensorPositionErrorsFromString(const QString& values, bool& ok);

    ProcessType _processType;
    EllipticityCorrection _ellipticityCorrection;
    InversionMethod _inversionMethod;
    int _rotateStepCount;
    int _maxPeakCount;
    double _maxPeakCountFactor;
    bool _exportAllFKGrids;
    bool _saveUndefinedEllipticities;
    bool _sortResultValues;
    double _kmin, _kmax;
    double _cacheGridStepFactor, _cacheGridStep;
    double _gridStepFactor, _gridStep, _gridSizeFactor, _gridSize, _smin, _smax;
    double _minAzimuth, _maxAzimuth;
    Curve<Point2D> _refineSlowMin, _refineSlowMax;
    double _minimumDistance, _maximumDistance;
    double _velocityRelativePrecision, _ellipticityAbsolutePrecision;
    double _absoluteThreshold, _relativeThreshold;
    double _damping;
    double _sourceGridStep, _sourceGridSize;
    QList<QStringList> _arrays;
    QString _fixedEllipticityFileName;
    QMap<QString, double> _sensorOrientationErrors;
    QMap<QString, Point2D> _sensorPositionErrors;
    double _sensorOrientationApertureAngle;
    double _sensorOrientationMaximumRadius;
    double _sensorOrientationMaximumCorrection;
    int _sensorOrientationMaximumIterations;
    int _brightSpots;
  };

  inline bool FKParameters::isRefine() const
  {
    return !_refineSlowMin.isEmpty() &&
        _refineSlowMax.count()==_refineSlowMin.count();
  }

#define FKPARAMETERS_PASSIVE_PROCESS_TYPES \
  case FKParameters::Conventional: \
  case FKParameters::Capon: \
  case FKParameters::Omni: \
  case FKParameters::ConventionalRadial: \
  case FKParameters::CaponRadial: \
  case FKParameters::ConventionalTransverse: \
  case FKParameters::CaponTransverse: \
  case FKParameters::ConventionalRayleigh: \
  case FKParameters::RTBF: \
  case FKParameters::ARTBF: \
  case FKParameters::RTBFFixedEll: \
  case FKParameters::ARDS: \
  case FKParameters::SensorOrientation: \
  case FKParameters::RDSSingle: \
  case FKParameters::PoggiVertical: \
  case FKParameters::PoggiRadial: \
  case FKParameters::LDS2: \
  case FKParameters::LDS3:

#define FKPARAMETERS_ACTIVE_PROCESS_TYPES \
  case FKParameters::ActiveConventional: \
  case FKParameters::ActiveCapon: \
  case FKParameters::ActiveConventionalRayleigh: \
  case FKParameters::ActiveRTBF:

} // namespace ArrayCore

#endif // FKPARAMETERS_H
