/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2023-01-06
**  Copyright: 2023
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ESACStatistics.h"

namespace ArrayCore {

  /*!
    \class ESACStatistics ESACStatistics.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  ESACStatistics::ESACStatistics()
  {
    TRACE;
    _nComp=0;
    _samples=nullptr;
    _opt.setRelativePrecision(1e-5);
  }

  /*!
    Description of destructor still missing
  */
  ESACStatistics::~ESACStatistics()
  {
    TRACE;
    delete [] _samples;
  }

  void ESACStatistics::setParameters(int nComponents)
  {
    TRACE;
    _nComp=nComponents;
    ASSERT(!_samples);
    _samples=new VectorList<Point2D>[_nComp];
  }

  /*!
    Marquardt (1963) least-square for non-linear functions.
  */
  double ESACStatistics::leastSquareWaveNumber(double k0, int iComp)
  {
    _currentComponent=iComp;
    Attributes a;
    a.x=k0;
    if(!_opt.minimize(a, this, 0.25*k0)) {
      _opt.curves<ESACStatistics, Attributes>(this, 0.1*k0, 10.0*k0, 0.01*k0);
      a.x=k0;
      _opt.minimize(a, this, 0.25*k0);
    }
    return a.x;
  }

#if 0
    double k=k0;
    const VectorList<Point2D>& samples=_samples[iComp];
    if(samples.count()>1) {
      printf("#\n");
      for(double k=0.1*k0; k<10*k0; k+=0.01*k0) {
        double s=0.0, si;
        for(int i=samples.count()-1; i>=0; i--) {
          const Point2D& p=samples.at(i);
          si=p.y()-j0(k*p.x());
          si*=si;
          s+=si;
        }
        printf("%lf %lf\n", k, s);
      }

      DoubleMatrix p(samples.count(), 1); // First derivative versus k
      DoubleMatrix y(samples.count(), 1); // Difference between data and model
      double lambda=1e-2;
      double delta;
      do {
        for(int i=samples.count()-1; i>=0; i--) {
          const Point2D& s=samples.at(i);
          double kr=k*s.x();
          p.at(i, 0)=s.x()*j1(kr);  // Derivative of J0=-J1 (Wolfram)
          y.at(i, 0)=s.y()-j0(kr);
        }
        double a=p.scalarProduct(p);
        double g=p.scalarProduct(y);
        // aStar is 1 in all cases
        /*double gStar=g/sqrt(a);
        double deltaTStar=gStar;
        double deltaStar=gStar/(1.0+lambda);
        double delta=deltaStar/sqrt(a);*/
        delta=g/(a*(1.0+lambda));
        k+=delta;
      } while(delta>1e-10*k);
    }
    return k;
  }
#endif

  void ESACStatistics::setFunctionValue(Attributes& a) const
  {
    const VectorList<Point2D>& samples=_samples[_currentComponent];
    double si, dsi, ddsi, kr, j0kr;
    a.value=0.0;
    a.slope=0.0;
    a.concavity=0.0;
    // Derivative of J0=-J1 (Wolfram)
    // Derivative of J1=0.5*(J0-J2) (Wolfram)
    //    Recurrence relation (Wikipedia)
    //     2/x J1(x)=J0(x)+J2(x)
    //    Hence, derivative of J1 becomes
    //     J0(x)-J1(x)/x
    for(int i=samples.count()-1; i>=0; i--) {
      const Point2D& p=samples.at(i);
      kr=a.x*p.x();
      j0kr=j0(kr);
      si=p.y()-j0kr;
      dsi=p.x()*j1(kr);
      ddsi=dsi*dsi+si*p.x()*(p.x()*j0kr-dsi/kr);
      dsi*=si;
      si*=si;
      a.value+=si;
      a.slope+=dsi;
      a.concavity+=ddsi;
    }
    a.slope*=2.0;
    a.concavity*=2.0;
  }

  void ESACStatistics::setFunctionDerivatives(Attributes&) const
  {
    // everything already calculated in setFunctionValue()
  }

  /*printf("#\n");
  for(double k=0.001*omega; k<0.01*omega; k+=0.0001*omega) {
    double s=0.0, si;
    for(int i=samples.count()-1; i>=0; i--) {
      const Point2D& p=samples.at(i);
      si=p.y()-j0(k*p.x());
      si*=si;
      s+=si;
    }
    printf("%lf %lf\n", k, s);
  }
  printf("#\n");
  for(double k=0.001*omega; k<0.01*omega; k+=0.0001*omega) {
    double s=0.0, si;
    for(int i=samples.count()-1; i>=0; i--) {
      const Point2D& p=samples.at(i);
      si=p.y()-j0(k*p.x());
      si*=p.x()*j1(k*p.x());
      s+=si;
    }
    printf("%lf %lf\n", k, 2.0*s);
  }*/

} // namespace ArrayCore

