/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-02-02
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ConvFKRayleighAll.h"
#include "FKCrossSpectrum.h"
#include "FKPower.h"
#include "ConvEllipticityEngine.h"

namespace ArrayCore {

//#define NEWTON_EVALUATION_STATS
#ifdef NEWTON_EVALUATION_STATS
  static qint64 nEvaluations[]={0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};
  static Mutex evaluationMutex;
#endif

  /*!
    \class ConvFKRayleighAll ConvFKRayleighAll.h
    \brief Rayleigh conventional FK power

    Rayleigh conventional FK power versus wavenumber (k) function.
    The cross-spectrum is already projected on the radial direction.

    Angular ellipticity steering is considered.
  */

  /*!
    Description of destructor still missing
  */
  ConvFKRayleighAll::~ConvFKRayleighAll()
  {
#ifdef NEWTON_EVALUATION_STATS
    App::log(tr("Number of Newton attribute evaluations:\n"));
    qint64 sum=0;
    for(int i=2; i<=20; i++) {
      sum+=nEvaluations[i];
    }
    qint64 cum=0;
    for(int i=2; i<=20; i++) {
      cum+=nEvaluations[i];
      App::log(tr("%1 %2 % %3 % %4\n")
               .arg(i)
               .arg(100.0*nEvaluations[i]/sum)
               .arg(100.0*cum/sum)
               .arg(nEvaluations[i]));
    }
#endif
  }

  void ConvFKRayleighAll::initialize(const FKParameters& param)
  {
    ConvFKRayleigh::initialize(param);
    ConvEllipticityEngine * eng;
    eng=new ConvEllipticityEngine(cache(), param);
    _ellEngine=eng;
  }

  inline void ConvFKRayleighAll::maximumEllipticity() const
  {
    // We admit values ouside -pi/2 to p/2 to avoid been trapped right at +/- pi/2
    // This happens if _xi is between the limits and the minimum
    _ellOptimization.maximize(_xi, this, M_PI/8.0);
#ifdef NEWTON_EVALUATION_STATS
    evaluationMutex.lock();
    if(_xi.evaluationCount>=20) {
      nEvaluations[20]++;
    } else {
      nEvaluations[_xi.evaluationCount]++;
    }
    evaluationMutex.unlock();
#endif
  }

  double ConvFKRayleighAll::value(const Vector<int>& index) const
  {
    if(isInsideLimits(index)) {
      oneComponentSteering(index);
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(index));
      maximumEllipticity();
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  double ConvFKRayleighAll::value(Vector<double>& kell) const
  {
    if(isInsideLimits(kell)) {
      oneComponentSteering(kell);
      powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));
      maximumEllipticity();
      canonicalEllipticity(_xi.x);
      kell[2]=_xi.x;
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  void ConvFKRayleighAll::setFunctionValue(Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    const Vector<Complex>& e=FKSteering::twoComponentRayleighCross(a.ell, powerEngine());
    powerEngine()->setConventionalValue(e);
    a.value=powerEngine()->value();
  }

  void ConvFKRayleighAll::setFunctionDerivatives(Attributes& a) const
  {
    // Get first derivative
    const Vector<Complex>& de=FKSteering::twoComponentRayleighCrossGradient(a.ell, powerEngine());
    a.slope=2.0*powerEngine()->factorN()*powerEngine()->ellipticityGradient(de);
    // Get second derivative
    a.concavity=powerEngine()->factorN()*powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(a.concavity-a.value);
  }

  /*!
    Calculates a noise factor from the second derivative of beam power
    and the second derivative of the beam pattern.
  */
  double ConvFKRayleighAll::wavenumberNormalizedConcavity(const Vector<double>& kell) const
  {
    oneComponentSteering(kell);
    powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));

    Complex ell;
    ell.setUnitExp(kell[2]);
    const Vector<Complex>& e0=FKSteering::twoComponentRayleighCross(ell, powerEngine());
    powerEngine()->setConventionalValue(e0);
    double p=powerEngine()->value();
    steering()->twoComponentSensorRadialProjections(kell, powerEngine());
    powerEngine()->conventionalKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
    return powerEngine()->conventionalKRadialSecondDerivative()/p;
  }

  /*!
    Analyse with

    cat /tmp/kderivatives | awk '{print $1, $2}' | gpcurve -derivative | gpcurve -derivative | figue -c
    cat /tmp/kderivatives | figue -c -y-columns 1,2
  */
  void ConvFKRayleighAll::testKDerivatives(const Vector<double>& kell) const
  {
    powerEngine()->setCrossSpectrum(_crossSpectrum->rotatedMatrix(kell));
    Complex ell;
    ell.setUnitExp(kell[2]);

    QFile f("/tmp/kderivatives");
    f.open(QIODevice::WriteOnly);
    QTextStream s(&f);

    PrivateVector<double> k1(kell);
    k1/=k1.length(0, 1);
    for(double k=0; k<6.28; k+=0.001) {
      PrivateVector<double> kv(k1);
      kv*=k;
      oneComponentSteering(kv);
      const Vector<Complex>& e0=FKSteering::twoComponentRayleighCross(ell, powerEngine());
      powerEngine()->setConventionalValue(e0);
      double p=powerEngine()->value();
      steering()->twoComponentSensorRadialProjections(kv, powerEngine());
      double dp=powerEngine()->conventionalKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
      double ddp=powerEngine()->conventionalKRadialSecondDerivative();
      s << k << " " << p << " " << dp << " " << ddp << Qt::endl;
    }
    f.close();
  }

} // namespace ArrayCore

