/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2018-10-04
**  Copyright: 2018-2019
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "ActiveConvFKRayleighAll.h"
#include "ActiveConvEllipticityEngine.h"

namespace ArrayCore {

  /*!
    \class ActiveConvFKRayleighAll ActiveConvFKRayleighAll.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  void ActiveConvFKRayleighAll::initialize(const FKParameters& param)
  {
    ActiveConvFKRayleigh::initialize(param);
    ActiveConvEllipticityEngine * eng;
    eng=new ActiveConvEllipticityEngine(cache(), param);
    eng->setCrossSpectrum(_crossSpectrum);
    _ellEngine=eng;
  }

  inline void ActiveConvFKRayleighAll::maximumEllipticity() const
  {
    // We admit values ouside -pi/2 to p/2 to avoid been trapped right at +/- pi/2
    // This happens if _xi is between the limits and the minimum
    _ellOptimization.maximize(_xi, this, M_PI/8.0);
  }

  double ActiveConvFKRayleighAll::value(Vector<double>& kell) const
  {
    kell[1]=0.0; // Only k[0] is modified by GridSearch
    if(isInsideLimits(kell)) {
      oneComponentSteering(kell[0]);
      maximumEllipticity();
      canonicalEllipticity(_xi.x);
      kell[2]=_xi.x;
      return _xi.value;
    } else {
      return -1.0;
    }
  }

  void ActiveConvFKRayleighAll::setFunctionValue(Attributes& a) const
  {
    a.ell.setUnitExp(a.x);
    const Vector<Complex>& e=FKSteering::twoComponentRayleighCross(a.ell, powerEngine());
    powerEngine()->setConventionalValue(e);
    a.value=powerEngine()->value();
  }

  void ActiveConvFKRayleighAll::setFunctionDerivatives(Attributes& a) const
  {
    // Get first derivative
    const Vector<Complex>& de=FKSteering::twoComponentRayleighCrossGradient(a.ell, powerEngine());
    a.slope=2.0*powerEngine()->factorN()*powerEngine()->ellipticityGradient(de);
    // Get second derivative
    a.concavity=powerEngine()->factorN()*powerEngine()->xiSecondDerivative();
    a.concavity=2.0*(a.concavity-a.value);
  }

  /*!
    Calculates a noise factor from the second derivative of beam power
    and the second derivative of the beam pattern.
  */
  double ActiveConvFKRayleighAll::wavenumberNormalizedConcavity(const Vector<double>& kell) const
  {
    oneComponentSteering(kell[0]);

    Complex ell;
    ell.setUnitExp(kell[2]);
    const Vector<Complex>& e0=FKSteering::twoComponentRayleighCross(ell, powerEngine());
    powerEngine()->setConventionalValue(e0);
    double p=powerEngine()->value();
    _steering->twoComponentSensorProjections(powerEngine());
    powerEngine()->conventionalKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
    return powerEngine()->conventionalKRadialSecondDerivative()/p;
  }

  /*!
    Analyse with

    cat /tmp/kderivatives | awk '{print $1, $2}' | gpcurve -derivative | gpcurve -derivative | figue -c
    cat /tmp/kderivatives | figue -c -y-columns 1,2
  */
  void ActiveConvFKRayleighAll::testKDerivatives(const Vector<double>& kell) const
  {
    Complex ell;
    ell.setUnitExp(kell[2]);

    QFile f("/tmp/kderivatives");
    f.open(QIODevice::WriteOnly);
    QTextStream s(&f);

    for(double k=0; k<6.28; k+=0.001) {
      oneComponentSteering(k);
      const Vector<Complex>& e0=FKSteering::twoComponentRayleighCross(ell, powerEngine());
      powerEngine()->setConventionalValue(e0);
      double p=powerEngine()->value();
      _steering->twoComponentSensorProjections(powerEngine());
      double dp=powerEngine()->conventionalKRadialFirstDerivative(powerEngine()->ellipticitySteeringVector());
      double ddp=powerEngine()->conventionalKRadialSecondDerivative();
      s << k << " " << p << " " << dp << " " << ddp << Qt::endl;
    }
    f.close();
    exit(0);
  }

} // namespace ArrayCore

