/***************************************************************************
**
**  This file is part of ArrayCore.
**
**  ArrayCore is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  ArrayCore is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2022-07-22
**  Copyright: 2022
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include "AbstractFKRayleigh.h"
#include "AbstractEllipticityEngine.h"

namespace ArrayCore {

  /*!
    \class AbstractFKRayleigh AbstractFKRayleigh.h
    \brief Brief description of class still missing

    Full description of class still missing
  */

  /*!
    Description of constructor still missing
  */
  AbstractFKRayleigh::AbstractFKRayleigh(FKCache * cache)
    : AbstractFKFunction(cache)
  {
    _xi.x=0.0;
    _ellEngine=nullptr;
    _ellipticityCorrection=FKParameters::ThreeComponent;
  }

  AbstractFKRayleigh::~AbstractFKRayleigh()
  {
    delete _ellEngine;
  }

  void AbstractFKRayleigh::initialize(const FKParameters& param)
  {
    _ellipticityCorrection=param.ellipticityCorrection();
  }

  void AbstractFKRayleigh::setAssociatedResults(const Vector<double>& kell, double power,
                                                FKPeaks::Value& val) const
  {
    val.power=power;
    if(_ellEngine) {
      switch(_ellipticityCorrection) {
      case FKParameters::None:
        break;
      case FKParameters::ThreeComponent:
        threeComponentEllipticity(kell, val);
        return;
      }
    }
    rawEllipticity(kell, val);
  }

  double AbstractFKRayleigh::radialPower(Vector<double>& kell) const
  {
    ASSERT(_ellEngine);
    return _ellEngine->radialPower(kell, _crossSpectrum);
  }

  double AbstractFKRayleigh::radialConcavity(Vector<double>& kell, int axis) const
  {
    ASSERT(_ellEngine);
    return _ellEngine->radialConcavity(kell, axis);
  }

  double AbstractFKRayleigh::verticalPower(Vector<double>& kell) const
  {
    ASSERT(_ellEngine);
    return _ellEngine->verticalPower(kell, _crossSpectrum);
  }

  double AbstractFKRayleigh::verticalConcavity(Vector<double>& kell, int axis) const
  {
    ASSERT(_ellEngine);
    return _ellEngine->verticalConcavity(kell, axis);
  }

  void AbstractFKRayleigh::rawEllipticity(const Vector<double>& kell, FKPeaks::Value& val) const
  {
    val.ellipticity=Angle::radiansToDegrees(kell[2]);
    val.verticalNoise=std::numeric_limits<double>::quiet_NaN();
    val.horizontalNoise=std::numeric_limits<double>::quiet_NaN();
  }

  void curveToFile(const QString& fileName, double eh, double ez,
                   const Polynomial& p)
  {
    QFile f(fileName);
    if(f.open(QIODevice::WriteOnly)) {
      QTextStream s(&f);
      double xih=atan(eh);
      double xiz=atan(ez);
      double dxi=(xiz-xih)*0.00001;
      xiz=fabs(xiz);
      for(double xi=xih; fabs(xi)<xiz; xi+=dxi) {
        double e=tan(xi);
        s << Angle::radiansToDegrees(xi) << " " << p.value(e) << Qt::endl;
      }
      f.close();
    }
  }

  void curveToFile(const QString& fileName, double eh, double ez,
                   const PolynomialFraction& fraction)
  {
    QFile f(fileName);
    if(f.open(QIODevice::WriteOnly)) {
      QTextStream s(&f);
      double xih=atan(eh);
      double xiz=atan(ez);
      double dxi=(xiz-xih)*0.00001;
      xiz=fabs(xiz);
      for(double xi=xih; fabs(xi)<xiz; xi+=dxi) {
        double e=tan(xi);
        s << Angle::radiansToDegrees(xi) << " " << fraction.value(e) << Qt::endl;
      }
      f.close();
    }
  }

  PolynomialFraction AbstractFKRayleigh::phiH(double eh, double ez, double ch)
  {
    PolynomialFraction phi(0, 2);
    phi.numerator().setCoefficient(0, eh*(ez-eh));
    phi.denominator().setCoefficient(0, -eh*ez);
    phi.denominator().setCoefficient(1, eh+ez);
    phi.denominator().setCoefficient(2, -1.0);
    phi/=ch;
    phi-=1.0;
    phi*=phi;
    return phi;
  }

  PolynomialFraction AbstractFKRayleigh::phiZ(double eh, double ez, double cz)
  {
    PolynomialFraction phi(2, 2);
    phi.numerator().setCoefficient(2, (ez-eh)/ez);
    phi.denominator().setCoefficient(0, -eh*ez);
    phi.denominator().setCoefficient(1, eh+ez);
    phi.denominator().setCoefficient(2, -1.0);
    phi/=cz;
    phi-=1.0;
    phi*=phi;
    return phi;
  }

  PolynomialFraction AbstractFKRayleigh::phiA(double eh, double ez, double ea, double ca)
  {
    PolynomialFraction phi(2, 2);
    double ezh=ez-eh;
    double eah=ea-eh;
    double ezah=eh*(ez-ea);
    phi.numerator().setCoefficient(0, ezah*ezah);
    phi.numerator().setCoefficient(1, 2.0*eah*ezah);
    phi.numerator().setCoefficient(2, eah*eah);
    phi.denominator().setCoefficient(0, -eh*ea*ea*ezh);
    phi.denominator().setCoefficient(1, (ea*ea+eh*ez)*ezh);
    phi.denominator().setCoefficient(2, -eh*ezh);
    phi.denominator()-=phi.numerator();
    phi/=ca;
    phi-=1.0;
    phi*=phi;
    return phi;
  }

  PolynomialFraction AbstractFKRayleigh::cH(double eh, double ez)
  {
    PolynomialFraction phi(0, 2);
    phi.numerator().setCoefficient(0, eh*(ez-eh));
    phi.denominator().setCoefficient(0, -eh*ez);
    phi.denominator().setCoefficient(1, eh+ez);
    phi.denominator().setCoefficient(2, -1.0);
    return phi;
  }

  PolynomialFraction AbstractFKRayleigh::cZ(double eh, double ez)
  {
    PolynomialFraction phi(2, 2);
    phi.numerator().setCoefficient(2, (ez-eh)/ez);
    phi.denominator().setCoefficient(0, -eh*ez);
    phi.denominator().setCoefficient(1, eh+ez);
    phi.denominator().setCoefficient(2, -1.0);
    return phi;
  }

  PolynomialFraction AbstractFKRayleigh::cA(double eh, double ez, double ea)
  {
    PolynomialFraction phi(2, 2);
    double ezh=ez-eh;
    double eah=ea-eh;
    double ezah=eh*(ez-ea);
    phi.numerator().setCoefficient(0, ezah*ezah);
    phi.numerator().setCoefficient(1, 2.0*eah*ezah);
    phi.numerator().setCoefficient(2, eah*eah);
    phi.denominator().setCoefficient(0, -eh*ea*ea*ezh);
    phi.denominator().setCoefficient(1, (ea*ea+eh*ez)*ezh);
    phi.denominator().setCoefficient(2, -eh*ezh);
    phi.denominator()-=phi.numerator();
    return phi;
  }

  Polynomial AbstractFKRayleigh::phitH(double eh, double ez, double th)
  {
    Polynomial phi(2);
    phi.setCoefficient(0, -eh*(eh+th*(ez-eh)));
    phi.setCoefficient(1, eh+ez);
    phi.setCoefficient(2, -1.0);
    phi/=eh*(ez-eh)*th;
    phi*=phi;
    return phi;
  }

  PolynomialFraction AbstractFKRayleigh::phitZ(double eh, double ez, double tz)
  {
    PolynomialFraction phi(2, 2);
    phi.numerator().setCoefficient(0, -ez*ez*eh);
    phi.numerator().setCoefficient(1, ez*(eh+ez));
    phi.numerator().setCoefficient(2, -eh-tz*(ez-eh));
    phi.denominator().setCoefficient(2, 1.0);
    phi/=(ez-eh)*tz;
    phi*=phi;
    return phi;
  }

  PolynomialFraction AbstractFKRayleigh::phitA(double eh, double ez, double ea, double ta)
  {
    PolynomialFraction phi(2, 2);
    double eah=ea-eh;
    double ezah=eh*(ez-ea);
    phi.numerator().setCoefficient(0, -eh*ea*ea);
    phi.numerator().setCoefficient(1, ea*ea+eh*ez);
    phi.numerator().setCoefficient(2, -eh);
    phi.denominator().setCoefficient(0, ezah*ezah);
    phi.denominator().setCoefficient(1, 2.0*eah*ezah);
    phi.denominator().setCoefficient(2, eah*eah);
    phi*=(ez-eh)/ta;
    phi-=1.0;
    phi*=phi;
    return phi;
  }

  double AbstractFKRayleigh::bestRoot(const VectorList<double>& roots, const PolynomialFraction& phi, double phim)
  {
    int n=roots.count();
    double bestXi=std::numeric_limits<double>::quiet_NaN();
    double bestPhi[2];
    bestPhi[0]=std::numeric_limits<double>::infinity();
    bestPhi[1]=std::numeric_limits<double>::infinity();
    for(int i=0; i<n; i++) {
      double xi=roots.at(i);
      double phiValue=phi.value(tan(xi));
      if(phiValue>0.0) {
        if(phiValue<bestPhi[0]) {
          bestXi=xi;
          bestPhi[1]=bestPhi[0];
          bestPhi[0]=phiValue;
        } else if(phiValue<bestPhi[1]) {
          bestPhi[1]=phiValue;
        }
      }
    }
    if(isnormal(bestPhi[0])) {
      if(isnormal(bestPhi[1])) {
        if(bestPhi[0]*phim<bestPhi[1]) {
          return bestXi;
        } else {
          return std::numeric_limits<double>::quiet_NaN();
        }
      } else { // Usually at ez and eh, if not properly discarded, numerical errors may occur with negative values
               // of a theoretically positive function.
               // There is a single "normal" root, accept it.
        ::abort();
        return bestXi;
      }
    } else {  // No regular root
      return std::numeric_limits<double>::quiet_NaN();
    }
  }

  double AbstractFKRayleigh::solve(const PolynomialFraction& phi, int& rootCount,
                                   const double& eh, const double& ez, double phim) const
  {
    PolynomialFraction phiDer(phi);
    phiDer.derivate();

    VectorList<double> roots=phiDer.numerator().realRoots();
    // Convert to degrees to remove multiple solutions below 0.0001 deg precision
    // There are two poles at eh and ez. They are automatically discarded when
    // searching for the root of the numerator with the lowest phi value.
    for(int i=roots.count()-1; i>=0; i--) {
      double e=roots.at(i);
      if((eh<0.0 && ez<e && e<eh) || (eh>0.0 && eh<e && e<ez)) {
        roots[i]=Angle::degreesToRadians(Number::round(Angle::radiansToDegrees(atan(e)), 0.0001));
      } else {
        roots.removeAt(i);
      }
    }
    unique(roots);
    rootCount=roots.count();
    switch(rootCount) {
    case 0:  // bias in experimental values, impossible to find a root between eh and ez
      return std::numeric_limits<double>::quiet_NaN();
    case 1:
      rootCount=1;
      return roots.at(0);
    case 2:
    case 3:
    case 4:
    case 5:
    case 6: // 6 roots may occur if precision makes that poles are not eliminated by above conditions
      return bestRoot(roots, phi, phim);
    default: // Normally impossible because the polynomial is of degree 6
      showRoots(roots, phi, phiDer, eh, ez);
      roots=phiDer.numerator().realRoots();
      showRoots(roots, phi, phiDer, eh, ez);
      exit(2);
      return std::numeric_limits<double>::quiet_NaN();
    }
  }

  void AbstractFKRayleigh::showRoots(const VectorList<double>& roots,
                                     const PolynomialFraction& phi, const PolynomialFraction& phiDer,
                                     double eh, double ez)
  {
    App::log(tr("Found %1 rounded roots for phi' (xih=%2 deg, xiz=%3 deg)\n")
             .arg(roots.count())
             .arg(Angle::radiansToDegrees(atan(eh)))
             .arg(Angle::radiansToDegrees(atan(ez))));
    for(int i=0; i<roots.count(); i++) {
      double xi=roots.at(i);
      double e=tan(xi);
      App::log(tr("  At %1 deg, derivative %2, phi %3\n")
               .arg(Angle::radiansToDegrees(xi))
               .arg(phiDer.numerator().value(e))
               .arg(phi.value(e)));
    }
  }

  double AbstractFKRayleigh::minimumConcavityFactorHZ(double eh, double ez)
  {
    return 4.0*eh/(ez-eh);
  }

  double AbstractFKRayleigh::minimumConcavityFactorA(double eh, double ez, double ea)
  {
    return 4.0*eh*ea/((ea-eh)*ea+eh*(ez-ea));
  }

  bool AbstractFKRayleigh::compareXir(const PhiSolution& s1, const PhiSolution& s2)
  {
    if(isnan(s1.xir)) {
      return false;
    }
    if(isnan(s2.xir)) {
      return true;
    }
    return s1.xir<s2.xir;
  }

//#define ELLIPTICITY_EXTRA_VALUES

  void AbstractFKRayleigh::threeComponentEllipticity(const Vector<double>& kell, FKPeaks::Value& val) const
  {
    double invBeamPatternConcavity=1.0/cache()->arrayPatternSecondDerivativeAtOrigin(kell);
    PrivateVector<double> kellh(kell);
    _ellEngine->radialPower(kellh, _crossSpectrum);
 #ifdef ELLIPTICITY_EXTRA_VALUES
    double xih=kellh[2];
 #endif
    double eh=tan(kellh[2]);
    double ch=_ellEngine->radialWavenumberNormalizedConcavity(kellh)*invBeamPatternConcavity;

    PrivateVector<double> kellz(kell);
    _ellEngine->verticalPower(kellz, _crossSpectrum);
#ifdef ELLIPTICITY_EXTRA_VALUES
    double xiz=kellz[2];
#endif
    double ez=tan(kellz[2]);
    double cz=_ellEngine->verticalWavenumberNormalizedConcavity(kellz)*invBeamPatternConcavity;

    double xia=kell[2];
    double ea=tan(xia);
    double ca=wavenumberNormalizedConcavity(kell)*invBeamPatternConcavity;

#ifdef ELLIPTICITY_EXTRA_VALUES
    val.extraValues=new PrivateVector<double>(10);
    /*
       10 [0] -> validity of signs, th, tz and ta
               -5: no root found in phi function
               -4: ch<0 and cz<0
               -3: negative noise after correcting ellipticity
               -2: eh, ea, ez order not correct
               -1: ellipticities do not have the same sign
                0: ca<0
                1: ch>0 and cz>0
                2: ch>0
                3: cz>0
       11 [1] -> number of roots
       12 [2] -> best phi if th, tz and ta are valid
                0: phiHZ
                1: phiHA
                2: phiZA
       13 [3] -> phi value at best minimum
       14 [4] -> original uncorrected ellipticity (xia)
       15 [5] -> xih
       16 [6] -> xiz
       17 [7] -> ch
       18 [8] -> cz
       19 [9] -> ca

    */

    val.extraValues->at(4)=Angle::radiansToDegrees(xia);
    val.extraValues->at(5)=xih;
    val.extraValues->at(6)=xiz;
    val.extraValues->at(7)=ch;
    val.extraValues->at(8)=cz;
    val.extraValues->at(9)=ca;
#endif

    if(ez*eh<0.0 || ez*ea<0.0) {      // All ellipticities must have the same sign
      val.ellipticity=std::numeric_limits<double>::quiet_NaN();
      val.horizontalNoise=std::numeric_limits<double>::quiet_NaN();
      val.verticalNoise=std::numeric_limits<double>::quiet_NaN();
#ifdef ELLIPTICITY_EXTRA_VALUES
      val.extraValues->at(0)=-1.0;
#endif
      return;
    }

    if(fabs(eh)>fabs(ea) || fabs(ea)>fabs(ez)) {      // |eh|<|ea|<|ez|
      val.ellipticity=std::numeric_limits<double>::quiet_NaN();
      val.horizontalNoise=std::numeric_limits<double>::quiet_NaN();
      val.verticalNoise=std::numeric_limits<double>::quiet_NaN();
#ifdef ELLIPTICITY_EXTRA_VALUES
      val.extraValues->at(0)=-2.0;
#endif
      return;
    }

    // ch, cz and ca must be positive
    if(ca<0.0) {
      val.ellipticity=std::numeric_limits<double>::quiet_NaN();
      val.horizontalNoise=std::numeric_limits<double>::quiet_NaN();
      val.verticalNoise=std::numeric_limits<double>::quiet_NaN();
#ifdef ELLIPTICITY_EXTRA_VALUES
      val.extraValues->at(0)=0.0;
#endif
      return;
    }

    double xir;
    PolynomialFraction phi(0, 0);
    int rootCount;

    if(ch>0.0) {
      if(cz>0.0) {
        phi=phiH(eh, ez, ch)+phiZ(eh, ez, cz);
        xir=solve(phi, rootCount, eh, ez, 2.0);
#ifdef ELLIPTICITY_EXTRA_VALUES
        val.extraValues->at(0)=1.0;
#endif
       } else {
        phi=phiH(eh, ez, ch)+phiA(eh, ez, ea, ca);
        xir=solve(phi, rootCount, eh, ez, 100.0);
#ifdef ELLIPTICITY_EXTRA_VALUES
        val.extraValues->at(0)=2.0;
#endif
      }
    } else {
      if(cz>0.0) {
        phi=phiZ(eh, ez, cz)+phiA(eh, ez, ea, ca);
        xir=solve(phi, rootCount, eh, ez, 100.0);
#ifdef ELLIPTICITY_EXTRA_VALUES
        val.extraValues->at(0)=3.0;
#endif
      } else {
        val.ellipticity=std::numeric_limits<double>::quiet_NaN();
        val.horizontalNoise=std::numeric_limits<double>::quiet_NaN();
        val.verticalNoise=std::numeric_limits<double>::quiet_NaN();
#ifdef ELLIPTICITY_EXTRA_VALUES
        val.extraValues->at(0)=-4.0;
#endif
        return;
      }
    }
#ifdef ELLIPTICITY_EXTRA_VALUES
    val.extraValues->at(1)=rootCount;
    val.extraValues->at(2)=0;
    val.extraValues->at(3)=phi.value(tan(xir));
#endif
    if(isnormal(xir)) {
      val.ellipticity=Angle::radiansToDegrees(xir);
      double cxir=cos(xir);
      double sxir=sin(xir);
      val.horizontalNoise=sxir*(cxir*ez-sxir);
      val.verticalNoise=cxir*(sxir/eh-cxir);
      if(val.horizontalNoise<0.0 || val.verticalNoise<0.0) {
        val.ellipticity=std::numeric_limits<double>::quiet_NaN();
        val.verticalNoise=std::numeric_limits<double>::quiet_NaN();
        val.horizontalNoise=std::numeric_limits<double>::quiet_NaN();
#ifdef ELLIPTICITY_EXTRA_VALUES
        val.extraValues->at(0)=-3.0;
#endif
      }
    } else {
      val.ellipticity=std::numeric_limits<double>::quiet_NaN();
      val.verticalNoise=std::numeric_limits<double>::quiet_NaN();
      val.horizontalNoise=std::numeric_limits<double>::quiet_NaN();
#ifdef ELLIPTICITY_EXTRA_VALUES
      val.extraValues->at(0)=-5.0;
#endif
    }
  }

} // namespace ArrayCore

