/***************************************************************************
**
**  This file is part of phaseit.
**
**  This file may be distributed and/or modified under the terms of the
**  GNU General Public License version 2 or 3 as published by the Free
**  Software Foundation and appearing in the file LICENSE.GPL included
**  in the packaging of this file.
**
**  This file is distributed in the hope that it will be useful, but WITHOUT
**  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
**  FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for
**  more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program. If not, see <http://www.gnu.org/licenses/>.
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2011-08-09
**  Authors:
**    Marc Wathelet (ISTerre, Grenoble, France)
**
***************************************************************************/

#include <GeopsyGui.h>

#include "GridResults.h"

/*!
  \class GridResults GridResults.h
  \brief Brief description of class still missing

  Full description of class still missing
*/

/*!
  Description of constructor still missing
*/
GridResults::GridResults(QWidget * parent)
  : GraphicSheetMenu(parent)
{
  TRACE;
  setGeometry(5, 80, 500, 500);
  setWindowTitle(tr("Grid results"));
  sheet()->pageSize().setOrientation(QPageLayout::Landscape);

  _comments=nullptr;
  _coherence=nullptr;
  _phase=nullptr;
  _amplitude=nullptr;
  _colorMapCoherence=new ColorMapWidget(nullptr);
  _colorMapCoherence->setPrintBottomMargin(0.0);
  _colorMapPhase=new ColorMapWidget(nullptr);
  _colorMapPhase->setPrintBottomMargin(0.0);
  _colorMapAmplitude=new ColorMapWidget(nullptr);
  _colorMapAmplitude->setPrintBottomMargin(0.0);
  _resultCount=0;
}

/*!
  Description of destructor still missing
*/
GridResults::~GridResults()
{
  TRACE;
  delete [] _comments;

  delete [] _coherence;
  delete [] _phase;
  delete [] _amplitude;
}

void GridResults::setAxis(AxisWindow * w)
{
  TRACE;
  w->xAxis()->setTitle(tr("Time"));
  w->xAxis()->setScaleType(Scale::AbsoluteTime);
  w->yAxis()->setFrequency();
  w->yAxis()->setRange(0.2, 20.0);
}

void GridResults::setResultCount(int n)
{
  TRACE;
  sheet()->setStatusBar(GeopsyGuiEngine::instance()->statusBar());
  double y=0.5;
  _resultCount=n;
  _comments=new TextEdit*[_resultCount];
  _coherence=new IrregularGrid2DPlot*[_resultCount];
  _phase=new IrregularGrid2DPlot*[_resultCount];
  _amplitude=new IrregularGrid2DPlot*[_resultCount];
  AxisWindow * w;
  for(int ig=0; ig<_resultCount; ig++) {
    GraphicObjectGroup * g=new GraphicObjectGroup;

    w=new AxisWindow;
    w->setObjectName("coherencePlot");
    setAxis(w);
    _coherence[ig]=new IrregularGrid2DPlot(w);
    _coherence[ig]->setSmooth(false);
    connect(_colorMapCoherence, SIGNAL(changed(ColorMap)),
            _coherence[ig], SLOT(setColorMap(const ColorMap&)));
    w->setGeometry(0.5, 1.0, 9.0, 6.0);
    g->addObject(w);

    w=new AxisWindow;
    w->setObjectName("phasePlot");
    setAxis(w);
    _phase[ig]=new IrregularGrid2DPlot(w);
    _phase[ig]->setSmooth(false);
    connect(_colorMapPhase, SIGNAL(changed(ColorMap)),
            _phase[ig], SLOT(setColorMap(const ColorMap&)));
    w->setGeometry(9.5, 1.0, 9.0, 6.0);
    g->addObject(w);

    w=new AxisWindow;
    w->setObjectName("amplitudePlot");
    setAxis(w);
    _amplitude[ig]=new IrregularGrid2DPlot(w);
    _amplitude[ig]->setSmooth(false);
    connect(_colorMapAmplitude, SIGNAL(changed(ColorMap)),
            _amplitude[ig], SLOT(setColorMap(const ColorMap&)));
    w->setGeometry(18.5, 1.0, 9.0, 6.0);
    g->addObject(w);

    _comments[ig]=new TextEdit;
    _comments[ig]->setObjectName("label");
    _comments[ig]->setAnchor(GraphicObject::BottomRight);
    _comments[ig]->setGeometry(27.5, 1.0, 4.0, 0.5);
    g->addObject(_comments[ig]);

    sheet()->addObject(g);
    g->resize();
    g->setPrintLeft(0.5);
    g->setPrintTop(y);
    g->updateGeometry();
    y+=6.5;
  }

  _colorMapCoherence->setObjectName("ColorMapAmplitude");
  _colorMapCoherence->generateColorScale(10, ColorPalette::McNamesClip);
  _colorMapCoherence->setNumberPrecision(0);
  _colorMapCoherence->setTitle("Cohence (%)");
  _colorMapCoherence->setValues(15.0, 95.0, SamplingParameters::Linear);
  _colorMapCoherence->setPrintXAnchor(1.5);
  _colorMapCoherence->setPrintYAnchor(y);
  _colorMapCoherence->setPrintWidth(8);
  _colorMapCoherence->setPrintHeight(1.2);
  sheet()->addObject(_colorMapCoherence);

  _colorMapPhase->setObjectName("ColorMapPhase");
  _colorMapPhase->generateColorScale(11, ColorPalette::ColdHot);
  _colorMapPhase->setNumberPrecision(0);
  _colorMapPhase->setTitle("Phase shift (%)");
  _colorMapPhase->setValues(-18.0, 18.0, SamplingParameters::Linear);
  _colorMapPhase->setPrintXAnchor(10.5);
  _colorMapPhase->setPrintYAnchor(y);
  _colorMapPhase->setPrintWidth(8);
  _colorMapPhase->setPrintHeight(1.2);
  sheet()->addObject(_colorMapPhase);

  _colorMapAmplitude->setObjectName("ColorMapAmplitude");
  _colorMapAmplitude->generateColorScale(11, ColorPalette::ColdHot);
  _colorMapAmplitude->setNumberPrecision(0);
  _colorMapAmplitude->setTitle("Spectral ratio");
  _colorMapAmplitude->setValues(0.5, 2.0, SamplingParameters::Log);
  _colorMapAmplitude->setScaleType(Scale::Log);
  _colorMapAmplitude->setMajorTicks(2.0);
  _colorMapAmplitude->setMinorTicks(0.5);
  _colorMapAmplitude->setPrintXAnchor(19.5);
  _colorMapAmplitude->setPrintYAnchor(y);
  _colorMapAmplitude->setPrintWidth(8);
  _colorMapAmplitude->setPrintHeight(1.2);
  sheet()->addObject(_colorMapAmplitude);
}

void GridResults::setName(int ig, const QString& name)
{
  TRACE;
  _comments[ig]->setText(name);
  _comments[ig]->update();
}

void GridResults::setTimeReference(int ig, const DateTime& t)
{
  TRACE;
  _coherence[ig]->graph()->xAxis()->setTimeReference(t);
  _phase[ig]->graph()->xAxis()->setTimeReference(t);
  _amplitude[ig]->graph()->xAxis()->setTimeReference(t);
}

void GridResults::setValues(const TaskManager * tm)
{
  TRACE;
  ASSERT(_resultCount==tm->stations()->count()-1);
  AxisWindow * w;
  for(int i=0; i<_resultCount; i++) {

    w=_coherence[i]->graph();
    const IrregularGrid2D& gcoherence=tm->coherenceGrid(i);
    _coherence[i]->setGrid(gcoherence);
    ColorMap& pcoherence=_coherence[i]->beginColorMapChange();
    pcoherence=_colorMapCoherence->colorMap();
    _coherence[i]->endColorMapChange();
    if(gcoherence.nx()>0 && gcoherence.ny()>0) {
      w->xAxis()->setRange(gcoherence.x(0), gcoherence.x(gcoherence.nx()-1));
      w->yAxis()->setRange(gcoherence.y(0), gcoherence.y(gcoherence.ny()-1));
    }

    w=_phase[i]->graph();
    const IrregularGrid2D& gphase=tm->phaseGrid(i);
    _phase[i]->setGrid(gphase);
    ColorMap& pphase=_phase[i]->beginColorMapChange();
    pphase=_colorMapPhase->colorMap();
    _phase[i]->endColorMapChange();
    _phase[i]->setGrid(gphase);
    if(gphase.nx()>0 && gphase.ny()>0) {
      w->xAxis()->setRange(gphase.x(0), gphase.x(gphase.nx()-1));
      w->yAxis()->setRange(gphase.y(0), gphase.y(gphase.ny()-1));
    }

    w=_amplitude[i]->graph();
    const IrregularGrid2D& gamplitude=tm->amplitudeGrid(i);
    _amplitude[i]->setGrid(gamplitude);
    ColorMap& pamplitude=_amplitude[i]->beginColorMapChange();
    pamplitude=_colorMapAmplitude->colorMap();
    _amplitude[i]->endColorMapChange();
    if(gamplitude.nx()>0 && gamplitude.ny()>0) {
      w->xAxis()->setRange(gamplitude.x(0), gamplitude.x(gamplitude.nx()-1));
      w->yAxis()->setRange(gamplitude.y(0), gamplitude.y(gamplitude.ny()-1));
    }
  }
}

void GridResults::setLayout(double pageHeight)
{
  TRACE;
  QList<GraphicObject *> objects;
  for(int i=0; i<_resultCount; i++) {
    objects.append(_coherence[i]->graph()->absoluteGroup());
  }
  GraphicObject::setLayout(objects, 0, pageHeight);
  double y=objects.last()->printBottom()+0.5;
  int pageIndex=qFloor(y/pageHeight);
  if(y+_colorMapCoherence->printHeight()>(pageIndex+1)*pageHeight) {
    pageIndex++;
    y=objects.first()->printTop()+pageIndex*pageHeight;
  }
  _colorMapCoherence->setPrintTop(y);
  _colorMapCoherence->updateGeometry();
  _colorMapPhase->setPrintTop(y);
  _colorMapPhase->updateGeometry();
  _colorMapAmplitude->setPrintTop(y);
  _colorMapAmplitude->updateGeometry();
}
