/***************************************************************************
**
**  This file is part of gpmaplayer.
**
**  gpmaplayer is free software: you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation, either version 3 of the License, or
**  (at your option) any later version.
**
**  gpmaplayer is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with Foobar.  If not, see <http://www.gnu.org/licenses/>
**
**  See http://www.geopsy.org for more information.
**
**  Created: 2016-03-25
**  Copyright: 2016-2019

**
***************************************************************************/

#include <GeopsyCore.h>
#include <SciFigs.h>

#include "ImageSource.h"
#include "RowImageMerger.h"
#include "ColumnImageMerger.h"
#include "gpmaplayerVersion.h"
#include "gpmaplayerInstallPath.h"

PACKAGE_INFO("gpmaplayer", GPMAPLAYER)

ApplicationHelp * help();

// TODO: zoom19 & zoom20 ne fonctionnent pas

int main(int argc, char ** argv)
{
  Application a(argc, argv, help);

  SciFigsGlobal s;
  GeopsyCoreEngine e;

  // Options
  QString outputBaseName="map";
  UtmZone utmZone;
  int zoom=16;
  double radius=1000;
  Point2D * localization=0;

  // Check arguments
  int i, j = 1;
  for (i=1; i<argc; i++) {
    QByteArray arg = argv[i];
    if (arg[0]=='-') {
      if (arg=="-o") {
        Application::checkOptionArg(i, argc, argv);
        outputBaseName=argv[i];
      } else if (arg=="-utm-zone") {
        CoreApplication::checkOptionArg(i, argc, argv);
        utmZone.fromString(argv[i]);
        if(!utmZone.isValid()) {
          App::log(tr("gpmaplayer: %1\n").arg(UtmZone::parsingErrorMessage(argv[i])) );
          return false;
        }
      } else if (arg=="-z") {
        Application::checkOptionArg(i, argc, argv);
        zoom=CoreApplication::toInt(i, i-1, argv);
        if(zoom<14) {
          zoom=14;
          App::log(tr("gpmaplayer: zoom set to 14\n") );
        }
      } else if (arg=="-r") {
        Application::checkOptionArg(i, argc, argv);
        radius=CoreApplication::toDouble(i, i-1, argv);
      } else if (arg=="-f") {
        Application::checkOptionArg(i, argc, argv);
        ImageMerger::setFilterFrequency(1.0/CoreApplication::toDouble(i, i-1, argv));
      } else if (arg=="-dy") {
        Application::checkOptionArg(i, argc, argv);
        ImageMerger::setDefaultYStep(CoreApplication::toDouble(i, i-1, argv));
      } else if (arg=="-force-dy") {
        Application::checkOptionArg(i, argc, argv);
        ImageMerger::setYStep(CoreApplication::toDouble(i, i-1, argv));
      } else if (arg=="-l") {
        delete localization;
        localization=new Point2D;
        Application::checkOptionArg(i, argc, argv);
        localization->setX(CoreApplication::toDouble(i, i-1, argv));
        Application::checkOptionArg(i, argc, argv);
        localization->setY(CoreApplication::toDouble(i, i-2, argv));
      } else if (arg=="-debug-image") {
        ImageMerger::setDebugMode(true);
      } else if (arg=="-corr-row") {
        QImage im1, im2;
        Application::checkOptionArg(i, argc, argv);
        im1.load(argv[i]);
        Application::checkOptionArg(i, argc, argv);
        im2.load(argv[i]);
        RowImageMerger m;
        m.setImage(im1, 600, 1000);
        m.setImage(im2, 600, 1000);
        return 0;
      } else if (arg=="-corr-column") {
        QImage im1, im2;
        Application::checkOptionArg(i, argc, argv);
        im1.load(argv[i]);
        Application::checkOptionArg(i, argc, argv);
        im2.load(argv[i]);
        ColumnImageMerger m(0, 0, 0, 0);
        m.setImage(im1, 600, 1000);
        m.setImage(im2, 600, 1000);
        return 0;
      } else if (arg=="-C") {
        Application::checkOptionArg(i, argc, argv);
        if(!QDir::setCurrent(argv[i])) {
          App::log(tr("gpmaplayer: cannot set working directory to '%1''\n").arg(argv[i]) );
          return 2;
        }
      } else {
        App::log(tr("gpmaplayer: bad option %1, see -help\n").arg(argv[i]) );
        return 2;
      }
    } else {
      argv[j++]=argv[i];
    }
  }
  if(j<argc) {
    argv[j]=nullptr;
    argc=j;
  }

  if(!localization) {
    App::log(tr("gpmaplayer: localization not specified, see -help\n") );
    return 2;
  }

  ImageSource src;

  src.setUtmZone(utmZone);
  src.setZoom(zoom);
  src.setRect(ImageSource::rect(*localization, radius));
  src.setOutputBaseName(outputBaseName);
  delete localization;
  return a.exec();
}

ApplicationHelp * help()
{
  TRACE;
  ApplicationHelp * h=new ApplicationHelp;
  h->setOptionSummary( "[OPTIONS]");
  h->setComments("Download hybrid maps from Google service and transform them as a georeferenced image stored as .layer. "
                 "Coordinate system is always UTM. If the area of interest is close to the limit of an UTM zone you can "
                 "force the zone with '-utm-zone' option.\n"
                 "Note that the Google map service is free for 25000 download over a period of 90 days. A large area with a high "
                 "zoom level may include a lot of individual tiles to be downloaded. ");
  h->addGroup("gpmaplayer", "gpmaplayer");
  h->addOption("-o <NAME>","Set base name for ouput files (.png and .layer).");
  h->addOption("-C <DIR>","Set working directory.");
  h->addOption("-z <VALUE>","Set zoom (integer from 14 to 21, default=16). Highest levels are not available everywhere.");
  h->addOption("-r <VALUE>","Set radius of area to be mapped in m (default=1000 m).");
  h->addOption("-l <LONG LAT>","Localization of the center of the area of interest (decimal degrees).");
  h->addOption("-f <WAVELENGTH>","High pass filter wavelength in pixels (default=200).");
  //h->addOption("-dx <STEP>","Default pixel step in X (when correlation<80%, default=0).");
  h->addOption("-dy <STEP>","Default pixel step in Y (when correlation<80%, default=0).");
  h->addOption("-force-dy <STEP>","Force Y step, no correlation is computed, use only to fix errors.");
  h->addOption("-utm-zone <ZONE>","Force the UTM zone with format 'XXY'' where XX is a number in [1, 60] and Y a letter in [C-H, J-N, P-X].");
  h->addOption("-debug-image","Save images and correlation signals for debug.");
  h->addOption("-corr-column <IM1 IM2>","Merge column images IM1 and IM2 (used for debug only).");
  h->addOption("-corr-row <IM1 IM2>","Merge row images IM1 and IM2 (used for debug only).");
  return h;
}
